<?php
namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\PermissionRegistrar;
use Illuminate\Support\Facades\Hash;

class RolePermissionSeeder extends Seeder
{
    public function run(): void
    {
        // Pastikan cache permissions/roles dibersihkan
        app()[PermissionRegistrar::class]->forgetCachedPermissions();

        // Normalisasi guard untuk role & permission yang mungkin terimport dari dump (gunakan 'web')
        \Spatie\Permission\Models\Role::query()->where('guard_name', '!=', 'web')->update(['guard_name' => 'web']);
        \Spatie\Permission\Models\Permission::query()->where('guard_name', '!=', 'web')->update(['guard_name' => 'web']);

        // Definisi role untuk aplikasi ujian (idempotent)
        $roleNames = ['superadmin', 'pegawai', 'guru', 'murid'];
        $roles = [];
        foreach ($roleNames as $name) {
            $roles[$name] = Role::firstOrCreate(['name' => $name, 'guard_name' => 'web']);
        }

        // Definisi permission untuk aplikasi ujian (idempotent)
        $permissions = [
            // Dashboard
            'view-dashboard',

            // Manajemen User
            'manage-users', 'create-user', 'edit-user', 'delete-user', 'view-user', 'reset-password',

            // Manajemen Role & Permission
            'manage-roles', 'assign-permissions', 'view-role', 'view-permission',

            // Manajemen Ujian
            'create-exam', 'edit-exam', 'delete-exam', 'view-exam', 'manage-exams',

            // Manajemen Soal
            'create-question', 'edit-question', 'delete-question', 'view-question', 'manage-questions',

            // Manajemen Mata Pelajaran (Subjects)
            'view-subject', 'create-subject', 'edit-subject', 'delete-subject', 'manage-subjects', 'view-all-subjects', 'view-own-subjects',

            // Akademik: Sekolah, Grade, Kelas, Peserta Ujian
            // Sekolah
            'view-school', 'create-school', 'edit-school', 'delete-school', 'manage-schools',
            // Grade
            'view-grade', 'create-grade', 'edit-grade', 'delete-grade', 'manage-grades',
            // Kelas
            'view-class', 'create-class', 'edit-class', 'delete-class', 'manage-classes',
            // Peserta Ujian
            'view-exam-participant', 'create-exam-participant', 'edit-exam-participant', 'delete-exam-participant', 'manage-exam-participants',
            'reset-exam-participant-password', 'import-exam-participants', 'export-exam-participants', 'export-exam-participants-cards-seats',

            // Manajemen Nilai
            'input-score', 'edit-score', 'view-score', 'manage-scores', 'export-scores',

            // Mengerjakan Ujian
            'take-exam', 'view-result',
        ];

        foreach ($permissions as $p) {
            Permission::firstOrCreate(['name' => $p, 'guard_name' => 'web']);
        }

        // Buat / perbarui user contoh untuk setiap role (idempotent)
        $superadmin = User::updateOrCreate(
            ['email' => 'superadmin@ujian.com'],
            ['name' => 'Super Admin', 'password' => Hash::make('Superadmin123!'), 'status' => 'aktif']
        );

        $pegawai = User::updateOrCreate(
            ['email' => 'pegawai@ujian.com'],
            ['name' => 'Pegawai TU', 'password' => Hash::make('Pegawai123!'), 'status' => 'aktif']
        );

        $guru = User::updateOrCreate(
            ['email' => 'guru@ujian.com'],
            ['name' => 'Guru Matematika', 'password' => Hash::make('Guru123!'), 'status' => 'aktif']
        );

        $murid = User::updateOrCreate(
            ['email' => 'murid@ujian.com'],
            ['name' => 'Murid', 'password' => Hash::make('Murid123!'), 'status' => 'aktif']
        );

        // Assign / sinkronkan role ke user (idempotent)
        $superadmin->syncRoles(['superadmin']);
        $pegawai->syncRoles(['pegawai']);
        $guru->syncRoles(['guru']);
        $murid->syncRoles(['murid']);

        // Assign / sinkronkan permission ke role (idempotent)
            foreach (Role::all() as $role) {
                if ($role->name === 'superadmin') {
                    // Superadmin memiliki semua permission
                    $role->syncPermissions(Permission::all());
                } elseif ($role->name === 'pegawai') {
                    // Pegawai dapat mengelola user dan melihat nilai
                    $role->syncPermissions([
                        'view-dashboard',
                        'manage-users', 'create-user', 'edit-user', 'view-user', 'reset-password',
                        'view-exam', 'view-score', 'export-scores',
                        // Subjects
                        'view-subject', 'view-all-subjects',
                        // Akademik (view-only)
                        'view-school', 'view-grade', 'view-class',
                        // Hak Akses (view-only)
                        'view-role', 'view-permission',
                        // Peserta Ujian (ops administrasi)
                        'view-exam-participant', 'create-exam-participant', 'edit-exam-participant', 'delete-exam-participant',
                        'reset-exam-participant-password', 'import-exam-participants', 'export-exam-participants', 'export-exam-participants-cards-seats',
                    ]);
                } elseif ($role->name === 'guru') {
                    // Guru dapat membuat ujian, soal, dan mengelola nilai
                    $role->syncPermissions([
                        'view-dashboard',
                        'view-user',
                        'create-exam', 'edit-exam', 'delete-exam', 'view-exam', 'manage-exams',
                        'create-question', 'edit-question', 'delete-question', 'view-question', 'manage-questions',
                        'input-score', 'edit-score', 'view-score', 'manage-scores', 'export-scores',
                        // Subjects: hanya mapel miliknya
                        'view-subject', 'view-own-subjects', 'create-subject', 'edit-subject', 'delete-subject',
                        // Akademik
                        'view-school', 'view-grade', 'view-class', 'edit-class',
                        // Peserta Ujian (view-only)
                        'view-exam-participant',
                    ]);
                } else {
                    // Murid hanya dapat mengerjakan ujian dan melihat hasil
                    $role->syncPermissions([
                        'view-dashboard',
                        'take-exam',
                        'view-result',
                    ]);
                }
            }
        }
}
