<?php

namespace App\Jobs;

use App\Models\ExamSession;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SaveExamDraftAnswers implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $sessionId;
    public $participantId;
    public $answers;
    public $meta;

    public function __construct(int $sessionId, int $participantId, array $answers, array $meta = [])
    {
        $this->sessionId = $sessionId;
        $this->participantId = $participantId;
        $this->answers = $answers;
        $this->meta = $meta;
        // Prioritaskan queue rendah agar tidak mengganggu submit final
        $this->onQueue('autosave');
    }

    public function handle(): void
    {
        $session = ExamSession::find($this->sessionId);
        if (!$session) {
            Log::warning('Autosave skipped: session not found', [
                'session_id' => $this->sessionId,
                'participant_id' => $this->participantId,
            ]);
            return;
        }
        if ((int) $session->exam_participant_id !== (int) $this->participantId) {
            Log::warning('Autosave skipped: participant mismatch', [
                'session_id' => $session->id,
                'participant_id' => $this->participantId,
            ]);
            return;
        }
        if ($session->finished_at) {
            Log::info('Autosave skipped: session already finished', [
                'session_id' => $session->id,
                'participant_id' => $this->participantId,
            ]);
            return;
        }

        DB::transaction(function () use ($session) {
            $locked = ExamSession::where('id', $session->id)->lockForUpdate()->first();
            if (!$locked) {
                return;
            }
            // Merge jawaban baru dengan draft yang ada dari baris yang di-lock
            $current = is_array($locked->draft_answers) ? $locked->draft_answers : [];
            $merged = $current;
            foreach ($this->answers as $qid => $opt) {
                $qidInt = (int) $qid;
                if ($qidInt <= 0) continue;

                // Null berarti hapus draft jawaban untuk soal ini
                if ($opt === null) {
                    unset($merged[$qidInt]);
                    continue;
                }

                // Hanya terima string; abaikan tipe lain
                if (!is_string($opt)) {
                    continue;
                }

                $trimmed = trim((string) $opt);
                if ($trimmed === '') {
                    // String kosong dianggap tidak ada jawaban
                    unset($merged[$qidInt]);
                    continue;
                }

                $upper = strtoupper($trimmed);
                if (in_array($upper, ['A','B','C','D','E'], true)) {
                    // Pilihan ganda A-E
                    $merged[$qidInt] = $upper;
                } else {
                    // Essay: simpan teks apa adanya (trimmed)
                    $merged[$qidInt] = $trimmed;
                }
            }

            $locked->draft_answers = $merged;
            $locked->save();

            Log::debug('Autosave draft answers', [
                'session_id' => $locked->id,
                'participant_id' => $this->participantId,
                'count' => count($this->answers),
            ]);
        });
    }
}