<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function index(Request $request) {
        // Ambil parameter paginasi dari request
        $perPage = (int) $request->input('per_page', 25);
        $page = (int) $request->input('page', 1);
        // Batasi nilai agar aman
        if ($perPage < 1) $perPage = 10;
        if ($perPage > 200) $perPage = 200;
        if ($page < 1) $page = 1;

        // Optional: basic search by name/email jika diperlukan ke depan
        $query = User::with('roles')->orderBy('id');
        // Aktifkan pencarian sederhana berdasarkan nama/email jika param `search` dikirim
        if ($request->filled('search')) {
            $search = trim((string)$request->input('search'));
            if ($search !== '') {
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%$search%")
                      ->orWhere('email', 'like', "%$search%");
                });
            }
        }

        return $query->paginate($perPage, ['*'], 'page', $page);
    }

    public function store(Request $r) {
        $r->validate([
            'name'=>'required',
            'email'=>'required|email|unique:users',
            'password'=>'nullable|string|min:6',
        ]);
        $user = User::create([
            'name' => $r->name,
            'email' => $r->email,
            'password' => isset($r->password) ? Hash::make($r->password) : Hash::make('12345678'),
            'nik'=>$r->nik, 'nuptk'=>$r->nuptk, 'terdaftar'=>$r->terdaftar,
            'jabatan'=>$r->jabatan, 'pegawai'=>$r->pegawai, 'status'=>$r->status ?? 'aktif'
        ]);
        if($r->role_id){
            $role = Role::find($r->role_id);
            if($role) $user->assignRole($role);
        }
        return response()->json($user->load('roles'));
    }

    public function show(User $user){ return response()->json($user->load('roles')); }

    public function update(Request $r, User $user){
        // Email menjadi opsional; jika dikirim, harus valid dan unik (kecuali diri sendiri)
        if ($r->has('email')) {
            $r->validate([ 'email' => "nullable|email|unique:users,email,{$user->id}" ]);
        }

        // Update hanya field yang dikirim agar tidak menimpa dengan null
        $payload = $r->only(['name','email','nik','nuptk','terdaftar','jabatan','pegawai','status']);
        // Jika email tidak dikirim, hapus dari payload supaya tidak mengubah yang ada
        if (!$r->has('email')) { unset($payload['email']); }

        $user->update($payload);

        if($r->role_id){
            $role = Role::find($r->role_id);
            $user->syncRoles($role);
        }
        return response()->json($user->load('roles'));
    }

    public function destroy(User $user){
        $user->tokens()->delete();
        $user->delete();
        return response()->json(['message'=>'User dihapus']);
    }

    public function resetPassword(User $user){
        $user->update(['password'=>Hash::make('12345678')]);
        return response()->json(['message'=>'Password direset']);
    }
}
