<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Subject;
use App\Models\Question;
use App\Models\Exam;
use App\Models\ExamSession;
use App\Models\ExamResult;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\RichText\RichText;

class SubjectController extends Controller
{
    // Middleware permission dipindahkan ke routes/api.php untuk menghindari error pada base Controller

    /**
     * Ekstrak path storage relatif (disk 'public') dari token gambar markdown
     * yang disisipkan pada teks atau opsi. Hanya memproses path yang berada di
     * bawah folder `questions/...`.
     */
    private function extractInlineImageStoragePaths(?string $content): array
    {
        if (!$content || !is_string($content)) return [];
        $paths = [];
        if (preg_match_all('/!\[[^\]]*\]\(([^\)]+)\)/', $content, $matches)) {
            foreach ($matches[1] as $rawUrl) {
                $url = trim((string)$rawUrl);
                $candidate = '';
                if ($url === '') continue;
                if (str_starts_with($url, 'http://') || str_starts_with($url, 'https://')) {
                    $path = parse_url($url, PHP_URL_PATH) ?: '';
                    $candidate = ltrim((string)$path, '/');
                } else {
                    $candidate = ltrim($url, '/');
                }
                $candidate = urldecode($candidate);
                if (str_starts_with($candidate, 'storage/')) {
                    $candidate = substr($candidate, strlen('storage/'));
                }
                if (str_starts_with($candidate, 'questions/')) {
                    $paths[] = $candidate;
                }
            }
        }
        return array_values(array_unique($paths));
    }

    public function index(Request $request)
    {
        $query = Subject::with(['owner:id,name'])->withCount('questions')->orderBy('name');
        if ($request->filled('id_school')) {
            $query->where('id_school', (int)$request->input('id_school'));
        }
        if ($request->filled('id_grade')) {
            $query->where('id_grade', (int)$request->input('id_grade'));
        }
        // Filter khusus: guru hanya melihat mapel yang dibuat oleh dirinya
        $user = $request->user();
        if ($user && $user->hasUserRole('guru')) {
            // Jika guru tidak memiliki izin view-all-subjects, batasi ke created_by
            if (!$user->hasPermission('view-all-subjects')) {
                $query->where('created_by', $user->id);
            }
        }
        $subjects = $query->get();
        return response()->json($subjects);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:50',
            'description' => 'nullable|string',
            'id_school' => 'required|exists:schools,id',
            'id_grade' => 'required|exists:grades,id',
        ]);

        $data = $validated;
        // Tetapkan pemilik mapel sebagai user yang login
        if ($request->user()) {
            $data['created_by'] = $request->user()->id;
        }
        $subject = Subject::create($data);
        return response()->json($subject, 201);
    }

    public function show(Subject $subject)
    {
        $subject->load(['owner:id,name']);
        $subject->loadCount('questions');
        return response()->json($subject);
    }

    public function update(Request $request, Subject $subject)
    {
        // Guru hanya boleh mengubah mapel miliknya
        $user = $request->user();
        if ($user && $user->hasUserRole('guru')) {
            if ($subject->created_by !== $user->id) {
                return response()->json(['message' => 'Anda tidak berhak mengubah mata pelajaran milik orang lain'], 403);
            }
        }
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:50',
            'description' => 'nullable|string',
            'id_school' => 'required|exists:schools,id',
            'id_grade' => 'required|exists:grades,id',
        ]);
        $subject->update($validated);
        return response()->json($subject);
    }

    public function destroy(Subject $subject)
    {
        // Guru hanya boleh menghapus mapel miliknya
        $user = request()->user();
        if ($user && $user->hasUserRole('guru')) {
            if ($subject->created_by !== $user->id) {
                return response()->json(['message' => 'Anda tidak berhak menghapus mata pelajaran milik orang lain'], 403);
            }
        }
        // Hapus semua file gambar soal yang terkait dan folder mapel
        try {
            $subjectSlug = Str::slug((string)($subject?->name ?? 'mapel'));
            $baseDir = 'questions/' . $subjectSlug;

            // Hapus file image pada setiap soal di subject ini (utama + inline)
            $questions = Question::where('subject_id', $subject->id)
                ->get(['id','image_path','text','option_a','option_b','option_c','option_d','option_e']);
            foreach ($questions as $q) {
                // Hapus gambar utama
                if ($q->image_path) {
                    Storage::disk('public')->delete($q->image_path);
                }
                // Hapus gambar inline dari teks dan opsi
                $all = [$q->text, $q->option_a, $q->option_b, $q->option_c, $q->option_d, $q->option_e];
                $toDelete = [];
                foreach ($all as $s) {
                    foreach ($this->extractInlineImageStoragePaths($s) as $p) {
                        $toDelete[$p] = true;
                    }
                }
                foreach (array_keys($toDelete) as $p) {
                    Storage::disk('public')->delete($p);
                }
            }
            // Hapus folder subject jika ada (ini juga menghapus sisa inline images baru)
            if (Storage::disk('public')->exists($baseDir)) {
                Storage::disk('public')->deleteDirectory($baseDir);
            }
        } catch (\Throwable $e) {
            // Lanjutkan penghapusan subject meskipun terjadi error pada file system
        }

        $subject->delete();
        return response()->json(['message' => 'Deleted']);
    }

    /**
     * Hapus banyak mata pelajaran (opsional: filter berdasarkan sekolah/grade) beserta seluruh
     * soal dan gambar terkait, serta ujian yang terkait dengan subject tersebut.
     * Endpoint ini untuk superadmin.
     */
    public function destroyAll(Request $request)
    {
        $idSchool = $request->input('id_school');
        $idGrade = $request->input('id_grade');

        return DB::transaction(function () use ($idSchool, $idGrade) {
            $subjectQuery = Subject::query();
            if (!empty($idSchool)) {
                $subjectQuery->where('id_school', (int)$idSchool);
            }
            if (!empty($idGrade)) {
                $subjectQuery->where('id_grade', (int)$idGrade);
            }

            $subjects = $subjectQuery->get();
            if ($subjects->isEmpty()) {
                return response()->json([
                    'deleted_subjects' => 0,
                    'deleted_questions' => 0,
                    'deleted_exams' => 0,
                    'deleted_sessions' => 0,
                    'deleted_results' => 0,
                    'message' => 'Tidak ada mata pelajaran yang cocok untuk dihapus'
                ]);
            }

            $subjectIds = $subjects->pluck('id')->all();

            // Hapus file gambar untuk setiap subject terlebih dahulu
            foreach ($subjects as $subject) {
                try {
                    $subjectSlug = Str::slug((string)($subject?->name ?? 'mapel'));
                    $baseDir = 'questions/' . $subjectSlug;

                    $questions = Question::where('subject_id', $subject->id)
                        ->get(['id','image_path','text','option_a','option_b','option_c','option_d','option_e']);
                    foreach ($questions as $q) {
                        if ($q->image_path) {
                            Storage::disk('public')->delete($q->image_path);
                        }
                        $all = [$q->text, $q->option_a, $q->option_b, $q->option_c, $q->option_d, $q->option_e];
                        $toDelete = [];
                        foreach ($all as $s) {
                            foreach ($this->extractInlineImageStoragePaths($s) as $p) {
                                $toDelete[$p] = true;
                            }
                        }
                        foreach (array_keys($toDelete) as $p) {
                            Storage::disk('public')->delete($p);
                        }
                    }
                    if (Storage::disk('public')->exists($baseDir)) {
                        Storage::disk('public')->deleteDirectory($baseDir);
                    }
                } catch (\Throwable $e) {
                    // Abaikan error filesystem agar proses DB tetap berjalan
                }
            }

            // Hapus ujian terkait subject ini (beserta sesi & hasilnya)
            $examIds = Exam::whereIn('id_subject', $subjectIds)->pluck('id')->all();
            $deletedResults = 0;
            $deletedSessions = 0;
            $deletedExams = 0;
            if (!empty($examIds)) {
                $sessionIds = ExamSession::whereIn('exam_id', $examIds)->pluck('id')->all();
                if (!empty($sessionIds)) {
                    $deletedResults = ExamResult::whereIn('exam_session_id', $sessionIds)->delete();
                    $deletedSessions = ExamSession::whereIn('id', $sessionIds)->delete();
                }
                $deletedExams = Exam::whereIn('id', $examIds)->delete();
            }

            // Hapus soal di semua subject tersebut
            $deletedQuestions = Question::whereIn('subject_id', $subjectIds)->delete();

            // Terakhir, hapus subject
            $deletedSubjects = Subject::whereIn('id', $subjectIds)->delete();

            return response()->json([
                'deleted_subjects' => $deletedSubjects,
                'deleted_questions' => $deletedQuestions,
                'deleted_exams' => $deletedExams,
                'deleted_sessions' => $deletedSessions,
                'deleted_results' => $deletedResults,
                'message' => 'Berhasil menghapus mata pelajaran beserta soal, gambar, dan ujian terkait'
            ]);
        });
    }

    /**
     * Import soal dari file DOCX ke dalam subject.
     * Format dukungan:
     * 1. Baris nomor pertanyaan: "1." atau "1)" diikuti teks pertanyaan
     * 2. Opsi pilihan ganda: baris yang diawali "A.", "B.", "C.", "D.", "E." (E opsional)
     * 3. Kunci jawaban wajib: baris "Kunci:" atau "Jawaban:" diikuti huruf A-E (case-insensitive)
     * Jika opsi tidak ada, soal dianggap Essay, namun kunci tetap wajib.
     */
    public function importDocx(Request $request, Subject $subject)
    {
        // Hanya guru pemilik subject yang boleh impor, kecuali punya permission create-question
        $user = $request->user();
        if ($user && $user->hasUserRole('guru')) {
            if ($subject->created_by !== $user->id && !$user->hasPermission('create-question')) {
                return response()->json(['message' => 'Anda tidak berhak mengimpor soal ke mata pelajaran ini'], 403);
            }
        }

        $validated = $request->validate([
            'file' => 'required|file|mimes:docx|max:5120', // max 5MB
        ]);

        $file = $validated['file'];

        try {
            $text = $this->extractTextFromDocx($file->getRealPath());
            // Pre-processing: normalisasi dan pembersihan sebelum parsing regex
            $text = $this->normalizeText($text);
            $text = $this->cleanTextBeforeParsing($text);
        } catch (\Throwable $e) {
            return response()->json(['message' => 'Gagal membaca file DOCX: '.$e->getMessage()], 422);
        }

        $parsed = $this->parseQuestionsFromText($text);
        if (empty($parsed)) {
            return response()->json(['message' => 'Tidak ditemukan pertanyaan dengan format yang didukung'], 422);
        }

        // Validasi: kunci jawaban wajib, A-D opsi wajib ada, E opsional, dan cek konten berbahaya
        $invalid = [];
        $malicious = [];
        $missingOptions = [];

        
        foreach ($parsed as $q) {
            $key = strtoupper(trim((string)($q['key'] ?? '')));
            if ($key === '' || !in_array($key, ['A','B','C','D','E'], true)) {
                $invalid[] = $q['number'] ?? null;
            }
            // A-D wajib terisi; E boleh kosong
            foreach (['A','B','C','D'] as $req) {
                if (trim((string)($q[$req] ?? '')) === '') {
                    $missingOptions[] = $q['number'] ?? null;
                    break;
                }
            }
            $fields = [
                (string)($q['text'] ?? ''),
                (string)($q['A'] ?? ''),
                (string)($q['B'] ?? ''),
                (string)($q['C'] ?? ''),
                (string)($q['D'] ?? ''),
                (string)($q['E'] ?? ''),
            ];
            foreach ($fields as $f) {
                if ($f !== '' && $this->containsMalicious($f)) {
                    $malicious[] = $q['number'] ?? null;
                    break;
                }
            }
        }
        if (!empty($invalid)) {
            return response()->json(['message' => 'Kunci jawaban wajib dan harus A-E. Periksa nomor: '.implode(', ', array_filter($invalid, fn($n) => $n !== null))], 422);
        }
        if (!empty($missingOptions)) {
            return response()->json(['message' => 'Format opsi tidak lengkap (A-D wajib, E boleh kosong). Periksa nomor: '.implode(', ', array_filter($missingOptions, fn($n) => $n !== null))], 422);
        }
        if (!empty($malicious)) {
            return response()->json(['message' => 'Konten berbahaya terdeteksi pada nomor: '.implode(', ', array_filter($malicious, fn($n) => $n !== null)).'. Harap bersihkan skrip/SQL/path traversal dari teks/opsi.'], 422);
        }

        $created = 0;
        DB::beginTransaction();
        try {
            foreach ($parsed as $q) {
                Question::create([
                    'subject_id' => $subject->id,
                    'created_by' => $user ? $user->id : null,
                    'text' => $this->sanitizeText((string)($q['text'] ?? ''), true),
                    'option_a' => $this->sanitizeText((string)($q['A'] ?? ''), true),
                    'option_b' => $this->sanitizeText((string)($q['B'] ?? ''), true),
                    'option_c' => $this->sanitizeText((string)($q['C'] ?? ''), true),
                    'option_d' => $this->sanitizeText((string)($q['D'] ?? ''), true),
                    'option_e' => $this->sanitizeText((string)($q['E'] ?? ''), true),
                    'key_answer' => $q['key'] ?? '',
                ]);
                $created++;
            }
            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['message' => 'Gagal menyimpan soal: '.$e->getMessage()], 500);
        }

        return response()->json([
            'message' => 'Berhasil mengimpor soal',
            'imported' => $created,
        ]);
    }

    /**
     * Unduh template DOCX format soal (contoh 1-2) untuk memudahkan pengguna.
     * Template berisi contoh struktur pertanyaan beserta opsi dan kunci.
     * Catatan: sistem menerima tanda baca (., ?, !, ...) dalam teks soal dan opsi.
     */
    public function importDocxTemplate(Request $request)
    {
        $tmp = tempnam(sys_get_temp_dir(), 'tpl_docx_');
        $zip = new \ZipArchive();
        if ($zip->open($tmp, \ZipArchive::OVERWRITE) !== true) {
            return response()->json(['message' => 'Gagal membuat file template'], 500);
        }
        $contentTypes = <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">
  <Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>
  <Default Extension="xml" ContentType="application/xml"/>
  <Override PartName="/word/document.xml" ContentType="application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml"/>
</Types>
XML;
        $rels = <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">
  <Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="word/document.xml"/>
</Relationships>
XML;

        // Template dengan format BOLD untuk elemen tetap dan normal untuk konten yang bisa diubah
        $paras = '';
        for ($i = 1; $i <= 2; $i++) {
            // Nomor soal (BOLD) + teks pertanyaan (normal)
            $paras .= "<w:p>";
            $paras .= "<w:r><w:rPr><w:b/></w:rPr><w:t>{$i}. </w:t></w:r>";
            $paras .= "<w:r><w:t>Teks pertanyaan {$i} di sini</w:t></w:r>";
            $paras .= "</w:p>";
            
            // Opsi A-E (huruf BOLD, teks normal)
            foreach (['A', 'B', 'C', 'D', 'E'] as $opt) {
                $paras .= "<w:p>";
                $paras .= "<w:r><w:rPr><w:b/></w:rPr><w:t>{$opt}. </w:t></w:r>";
                if ($opt === 'E') {
                    $paras .= "<w:r><w:t>(opsional - boleh kosong)</w:t></w:r>";
                } else {
                    $paras .= "<w:r><w:t>Teks opsi {$opt}</w:t></w:r>";
                }
                $paras .= "</w:p>";
            }
            
            // Kunci jawaban (label BOLD, jawaban normal)
            $paras .= "<w:p>";
            $paras .= "<w:r><w:rPr><w:b/></w:rPr><w:t>Kunci: </w:t></w:r>";
            $paras .= "<w:r><w:t>A</w:t></w:r>";
            $paras .= "</w:p>";
            
            // Spasi pemisah
            $paras .= "<w:p><w:r><w:t></w:t></w:r></w:p>";
        }
        $document = <<<XML
<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<w:document xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main">
  <w:body>
    {$paras}
    <w:sectPr/>
  </w:body>
  </w:document>
XML;

        $zip->addFromString('[Content_Types].xml', $contentTypes);
        $zip->addFromString('_rels/.rels', $rels);
        $zip->addFromString('word/document.xml', $document);
        $zip->close();

        return response()->download($tmp, 'template_soal.docx')->deleteFileAfterSend(true);
    }

    private function extractTextFromDocx(string $path): string
    {
        $zip = new \ZipArchive();
        if ($zip->open($path) !== true) {
            throw new \RuntimeException('Tidak bisa membuka file DOCX');
        }
        $xml = $zip->getFromName('word/document.xml');
        $zip->close();
        if ($xml === false) {
            throw new \RuntimeException('Dokumen DOCX tidak valid');
        }
        // Sederhanakan: ubah paragraf ke newline, tab ke spasi, strip tag XML
        $xml = preg_replace('/<w:tab\s*\/?>/i', '    ', $xml);
        $xml = preg_replace('/<\/?w:p[^>]*>/i', "\n", $xml);
        $xml = preg_replace('/<\/?w:br[^>]*>/i', "\n", $xml);
        $text = strip_tags($xml);
        // Normalisasi whitespace
        $text = preg_replace("/\r\n|\r|\n/", "\n", $text);
        $lines = array_map(function($l){ return trim((string)$l); }, explode("\n", $text));
        // Gabungkan multi-spaces
        $lines = array_map(function($l){ return preg_replace('/\s+/', ' ', $l); }, $lines);
        return implode("\n", array_filter($lines, fn($l) => $l !== ''));
    }

    // Normalisasi teks hasil ekstraksi DOCX sebelum parsing regex
    private function normalizeText(string $text): string
    {
        // Normalisasi newline
        $text = preg_replace("/\r\n|\r|\n/", "\n", $text);
        // Hilangkan karakter zero-width dan BOM
        $text = preg_replace('/[\x{200B}-\x{200D}\x{FEFF}]/u', '', $text);
        // Ganti NBSP ke spasi biasa
        $text = str_replace("\xC2\xA0", ' ', $text);
        $text = preg_replace('/\x{00A0}/u', ' ', $text);
        // Normalisasi kutip dan tanda baca umum
        $map = [
            '“' => '"', '”' => '"', '‘' => "'", '’' => "'",
            '–' => '-', '—' => '-', '•' => '',
        ];
        $text = strtr($text, $map);
        // Rapatkan pola nomor/huruf diikuti titik: "1 ." => "1."; "A ." => "A."
        $text = preg_replace('/\b(\d{1,4})\s*\.\s*/', '$1. ', $text);
        $text = preg_replace('/\b([A-Ea-e])\s*\.\s*/', '$1. ', $text);
        // Gabungkan multi-spasi
        $text = preg_replace('/[ \t]+/', ' ', $text);
        // Trim tiap baris
        $lines = array_map(fn($l) => trim($l), explode("\n", $text));
        return implode("\n", $lines);
    }

    // Pembersihan defensif sebelum parsing untuk cegah injeksi dan noise dokumen
    private function cleanTextBeforeParsing(string $text): string
    {
        // Buang karakter kontrol (kecuali newline)
        $text = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F]/u', ' ', $text);
        // Buang tag HTML jika ada residu
        $text = strip_tags($text);
        // Hilangkan skema berbahaya
        $text = preg_replace('/javascript\s*:/i', '', $text);
        $text = preg_replace('/data\s*:/i', '', $text);
        // Normalisasi newline dan trim akhir
        $text = preg_replace("/\r\n|\r|\n/", "\n", $text);
        $text = preg_replace('/\n{3,}/', "\n\n", $text);
        return trim($text);
    }

    private function parseQuestionsFromText(string $text): array
    {
        $lines = explode("\n", $text);
        $questions = [];
        $current = null;
        foreach ($lines as $line) {
            $l = trim($line);
            if ($l === '') continue;

            // Deteksi awal pertanyaan: WAJIB pola "nomor. teks" (titik langsung setelah nomor)
            if (preg_match('/^(\d{1,3})\.\s+(.+)$/', $l, $m)) {
                // Simpan yang lama
                if ($current) {
                    $questions[] = $current;
                }
                $current = [
                    'number' => (int)$m[1],
                    'text' => trim($m[2]),
                    'A' => '', 'B' => '', 'C' => '', 'D' => '', 'E' => '',
                    'key' => '',
                ];
                continue;
            }

            if (!$current) {
                // Abaikan baris sebelum pertanyaan pertama
                continue;
            }

            // Opsi A-E: pola kuat namun akomodatif terhadap spasi dan huruf kecil
            // Terima "A. ..." atau "a . ..."; teks opsi bisa kosong khusus untuk E
            if (preg_match('/^([A-Ea-e])\s*\.\s*(.*)$/', $l, $m)) {
                $opt = strtoupper($m[1]);
                $current[$opt] = trim($m[2]);
                continue;
            }

            // KUNCI: terima label "Kunci:" atau "Kunci jawaban:"; huruf A-E (case-insensitive)
            // Mengizinkan kutip di sekitar huruf dan tanda baca penutup opsional (., ), ])
            // Contoh valid: "Kunci: E", "Kunci: 'e'", "Kunci: E.", "Kunci jawaban: \"b\""
            if (preg_match('/^(kunci(?:\s+jawaban)?)\s*[:=\-]\s*["\']?([A-Ea-e])["\']?\s*[\.\)\]]?\s*$/i', $l, $m)) {
                $current['key'] = strtoupper(trim($m[2]));
                continue;
            }

            // Tambahkan ke teks pertanyaan jika bukan opsi/kunci
            $current['text'] = trim(($current['text'] ?? '') . ' ' . $l);
        }
        if ($current) $questions[] = $current;

        // Batasi maksimal 100 soal untuk menjaga performa
        if (count($questions) > 100) {
            $questions = array_slice($questions, 0, 100);
        }

        // Bersihkan trailing dash/empty
        $clean = [];
        foreach ($questions as $q) {
            $q['text'] = trim((string)$q['text']);
            foreach (['A','B','C','D','E'] as $k) {
                $q[$k] = trim((string)$q[$k]);
            }
            $q['key'] = strtoupper(trim((string)$q['key']));
            if ($q['text'] !== '') {
                $clean[] = $q;
            }
        }
        return $clean;
    }

    // Sanitasi dasar untuk mencegah konten berbahaya masuk ke DB
    private function sanitizeText(string $value, bool $allowBasicTags = true): string
    {
        $clean = preg_replace('/<script\b[^>]*>(.*?)<\/script>/is', '', $value);
        $clean = preg_replace('/on[a-z]+\s*=\s*"[^"]*"/i', '', $clean);
        $clean = preg_replace("/on[a-z]+\s*=\s*'[^']*'/i", '', $clean);
        $clean = preg_replace('/javascript\s*:/i', '', $clean);
        $clean = preg_replace('/data\s*:/i', '', $clean);
        $clean = preg_replace('/vbscript\s*:/i', '', $clean);
        $clean = preg_replace('/file\s*:/i', '', $clean);
        if ($allowBasicTags) {
            $clean = strip_tags($clean, '<b><strong><i><em><u><br><p><ul><ol><li><sup><sub>');
        } else {
            $clean = strip_tags($clean);
        }
        return trim($clean ?? '');
    }

    // Deteksi pola berbahaya pada teks
    private function containsMalicious(string $value): bool
    {
        $patterns = [
            '/<script\b/i',
            '/on[a-z]+\s*=/i',
            '/javascript\s*:/i',
            '/data\s*:/i',
            '/drop\s+table/i',
            '/alter\s+table/i',
            '/delete\s+from/i',
            '/truncate\s+table/i',
            '/update\s+\w+\s+set/i',
            '/insert\s+into/i',
            '/union\s+select/i',
            '/;\s*--/i',
            '/\.{2}\//', // ../
            '/\.{2}\\\\/', // ..\
        ];
        $patterns = array_merge($patterns, [
            '/vbscript\s*:/i',
            '/file\s*:/i',
            '/--\s*$/m',
            '/on[a-z]+\s*=\s*/i',
        ]);
        foreach ($patterns as $re) {
            if (preg_match($re, $value)) return true;
        }
        return false;
    }

    // Helper: ambil nilai sel berdasarkan indeks kolom dan baris (A1 style)
    private function getCellVal($sheet, int $columnIndex, int $rowIndex)
    {
        $coord = Coordinate::stringFromColumnIndex($columnIndex) . $rowIndex;
        $cell = $sheet->getCell($coord);
        if (!$cell) return null;
        // Gunakan nilai terhitung untuk menangani formula
        $val = $cell->getCalculatedValue();
        // Konversi RichText ke string bila perlu
        if ($val instanceof RichText) {
            $val = $val->getPlainText();
        }
        return $val;
    }

    // Normalisasi label header agar toleran terhadap variasi penamaan
    private function normalizeHeaderLabel(string $label): string
    {
        $l = strtolower(trim($label));
        // Ganti spasi dan tanda umum dengan underscore
        $l = str_replace([' ', '-', '.', ':', ';'], '_', $l);
        // Hapus karakter selain alnum dan underscore
        $l = preg_replace('/[^a-z0-9_]/', '', $l);
        // Rapikan underscore berulang
        $l = preg_replace('/_{2,}/', '_', $l);

        // Gabungkan variasi menjadi kunci kanonik
        switch ($l) {
            case 'soal':
            case 'question':
            case 'pertanyaan':
                return 'text';
            case 'optiona':
            case 'option_a':
            case 'opsi_a':
            case 'pilihan_a':
            case 'jawaban_a':
            case 'a':
                return 'option_a';
            case 'optionb':
            case 'option_b':
            case 'opsi_b':
            case 'pilihan_b':
            case 'jawaban_b':
            case 'b':
                return 'option_b';
            case 'optionc':
            case 'option_c':
            case 'opsi_c':
            case 'pilihan_c':
            case 'jawaban_c':
            case 'c':
                return 'option_c';
            case 'optiond':
            case 'option_d':
            case 'opsi_d':
            case 'pilihan_d':
            case 'jawaban_d':
            case 'd':
                return 'option_d';
            case 'optione':
            case 'option_e':
            case 'opsi_e':
            case 'pilihan_e':
            case 'jawaban_e':
            case 'e':
                return 'option_e';
            case 'kunci':
            case 'kunci_jawaban':
            case 'jawaban':
            case 'answer_key':
                return 'key';
            case 'no':
            case 'nomor':
            case 'number':
                return 'no';
        }
        return $l; // default kembalikan hasil normalisasi
    }

    private function findColumn(array $headers, array $candidates): ?int
    {
        foreach ($candidates as $c) {
            if (isset($headers[$c])) return $headers[$c];
        }
        return null;
    }

    /**
     * Unduh template Excel (.xlsx) untuk impor soal.
     * Header: No, Text, Option_A, Option_B, Option_C, Option_D, Option_E, Key
     */
    public function importExcelTemplate(Request $request)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['No','Text','Option_A','Option_B','Option_C','Option_D','Option_E','Key'];
        foreach ($headers as $i => $h) {
            $col = Coordinate::stringFromColumnIndex($i + 1);
            $sheet->setCellValue($col.'1', $h);
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        // Contoh 2 baris
        $sheet->setCellValue('A2', 1);
        $sheet->setCellValue('B2', 'Teks pertanyaan 1');
        $sheet->setCellValue('C2', 'Opsi A');
        $sheet->setCellValue('D2', 'Opsi B');
        $sheet->setCellValue('E2', 'Opsi C');
        $sheet->setCellValue('F2', 'Opsi D');
        $sheet->setCellValue('G2', '');
        $sheet->setCellValue('H2', 'A');

        $sheet->setCellValue('A3', 2);
        $sheet->setCellValue('B3', 'Teks pertanyaan 2');
        $sheet->setCellValue('C3', 'Opsi A');
        $sheet->setCellValue('D3', 'Opsi B');
        $sheet->setCellValue('E3', 'Opsi C');
        $sheet->setCellValue('F3', 'Opsi D');
        $sheet->setCellValue('G3', '');
        $sheet->setCellValue('H3', 'B');

        $writer = new Xlsx($spreadsheet);
        ob_start();
        $writer->save('php://output');
        $xlsx = ob_get_clean();

        return response($xlsx, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="template_soal.xlsx"',
        ]);
    }

    /**
     * Import soal dari Excel (xlsx/xls) ke Subject.
     * Mendukung header: No, Text, Option_A, Option_B, Option_C, Option_D, Option_E, Key
     */
    public function importExcel(Request $request, Subject $subject)
    {
        // Hanya guru pemilik subject yang boleh impor, kecuali punya permission create-question
        $user = $request->user();
        if ($user && $user->hasUserRole('guru')) {
            if ($subject->created_by !== $user->id && !$user->hasPermission('create-question')) {
                return response()->json(['message' => 'Anda tidak berhak mengimpor soal ke mata pelajaran ini'], 403);
            }
        }

        $validated = $request->validate([
            'file' => 'required|file|mimes:xlsx,xls|max:5120',
        ]);
        $file = $validated['file'];

        try {
            $spreadsheet = IOFactory::load($file->getRealPath());
            $sheet = $spreadsheet->getActiveSheet();
        } catch (\Throwable $e) {
            return response()->json(['message' => 'Gagal membaca file Excel: '.$e->getMessage()], 422);
        }

        // Baca header baris 1
        $headers = [];
        $highestColumn = $sheet->getHighestColumn();
        $highestColumnIndex = Coordinate::columnIndexFromString($highestColumn);
        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $raw = $this->getCellVal($sheet, $col, 1);
            $val = trim((string)($raw ?? ''));
            if ($val !== '') {
                $norm = $this->normalizeHeaderLabel($val);
                $headers[$norm] = $col;
            }
        }
        // Pemetaan fleksibel berdasarkan kandidat nama kolom
        $colText = $this->findColumn($headers, ['text']);
        $colKey  = $this->findColumn($headers, ['key']);
        $colA    = $this->findColumn($headers, ['option_a']);
        $colB    = $this->findColumn($headers, ['option_b']);
        $colC    = $this->findColumn($headers, ['option_c']);
        $colD    = $this->findColumn($headers, ['option_d']);
        $colE    = $this->findColumn($headers, ['option_e']);
        // Validasi header minimum
        if (!$colText || !$colKey) {
            return response()->json([
                'message' => 'Header Excel tidak valid. Wajib ada kolom Soal/Text dan Kunci/Key.'
            ], 422);
        }

        $map = [
            'No' => $this->findColumn($headers, ['no']),
            'Text' => $colText,
            'Option_A' => $colA,
            'Option_B' => $colB,
            'Option_C' => $colC,
            'Option_D' => $colD,
            'Option_E' => $colE,
            'Key' => $colKey,
        ];

        $created = 0;
        DB::beginTransaction();
        try {
            $row = 2;
            $maxRows = 1000; // batas aman
            while ($row <= $sheet->getHighestRow() && $created < $maxRows) {
                $text = trim((string)($this->getCellVal($sheet, $map['Text'], $row) ?? ''));
                $keyRaw = $this->getCellVal($sheet, $map['Key'], $row);
                $key = strtoupper(trim((string)($keyRaw ?? '')));
                // Terima angka 1-5 sebagai A-E
                if (preg_match('/^[1-5]$/', $key)) {
                    $letters = ['1' => 'A', '2' => 'B', '3' => 'C', '4' => 'D', '5' => 'E'];
                    $key = $letters[$key];
                }
                if ($text === '') { $row++; continue; }

                $optA = $map['Option_A'] ? (string)($this->getCellVal($sheet, $map['Option_A'], $row) ?? '') : '';
                $optB = $map['Option_B'] ? (string)($this->getCellVal($sheet, $map['Option_B'], $row) ?? '') : '';
                $optC = $map['Option_C'] ? (string)($this->getCellVal($sheet, $map['Option_C'], $row) ?? '') : '';
                $optD = $map['Option_D'] ? (string)($this->getCellVal($sheet, $map['Option_D'], $row) ?? '') : '';
                $optE = $map['Option_E'] ? (string)($this->getCellVal($sheet, $map['Option_E'], $row) ?? '') : '';

                // Sanitasi dasar
                $text = $this->sanitizeText($text);
                $optA = $this->sanitizeText($optA);
                $optB = $this->sanitizeText($optB);
                $optC = $this->sanitizeText($optC);
                $optD = $this->sanitizeText($optD);
                $optE = $this->sanitizeText($optE);

                // Simpan soal
                Question::create([
                    'text' => $text,
                    'option_a' => $optA,
                    'option_b' => $optB,
                    'option_c' => $optC,
                    'option_d' => $optD,
                    'option_e' => $optE,
                    'key_answer' => in_array($key, ['A','B','C','D','E']) ? $key : '',
                    'image_path' => null,
                    'created_by' => $user?->id,
                    'subject_id' => $subject->id,
                ]);
                $created++;
                $row++;
            }
            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['message' => 'Gagal menyimpan soal dari Excel: '.$e->getMessage()], 500);
        }

        return response()->json([
            'message' => 'Berhasil mengimpor soal dari Excel',
            'imported' => $created,
        ]);
    }
}