<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Score;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Symfony\Component\HttpFoundation\StreamedResponse;

class ScoreController extends Controller
{
    public function index(Request $request)
    {
        $query = Score::query()->orderByDesc('id');
        if ($request->filled('id_pelajaran')) {
            $query->where('id_pelajaran', (int)$request->input('id_pelajaran'));
        }
        if ($request->filled('tahun')) {
            $query->where('tahun', (string)$request->input('tahun'));
        }
        if ($request->filled('nisn')) {
            $query->where('nisn', (string)$request->input('nisn'));
        }
        return response()->json($query->get());
    }

    public function show(Score $score)
    {
        return response()->json($score);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nisn' => 'nullable|string|max:50',
            'nama' => 'nullable|string|max:255',
            'id_pelajaran' => 'nullable|exists:subjects,id',
            'name_pelajaran' => 'nullable|string|max:255',
            'sts_ganjil' => 'nullable|numeric',
            'sas' => 'nullable|numeric',
            'sts_genap' => 'nullable|numeric',
            'sat' => 'nullable|numeric',
            'us' => 'nullable|numeric',
            'harian' => 'nullable|numeric',
            'keterampilan' => 'nullable|numeric',
            'keterangan' => 'nullable|string',
            'teacher' => 'nullable|string|max:255',
            'tahun' => 'nullable|string|max:50',
        ]);

        $score = Score::create($validated);
        return response()->json($score, 201);
    }

    public function update(Request $request, Score $score)
    {
        $validated = $request->validate([
            'nisn' => 'nullable|string|max:50',
            'nama' => 'nullable|string|max:255',
            'id_pelajaran' => 'nullable|exists:subjects,id',
            'name_pelajaran' => 'nullable|string|max:255',
            'sts_ganjil' => 'nullable|numeric',
            'sas' => 'nullable|numeric',
            'sts_genap' => 'nullable|numeric',
            'sat' => 'nullable|numeric',
            'us' => 'nullable|numeric',
            'harian' => 'nullable|numeric',
            'keterampilan' => 'nullable|numeric',
            'keterangan' => 'nullable|string',
            'teacher' => 'nullable|string|max:255',
            'tahun' => 'nullable|string|max:50',
        ]);

        $score->update($validated);
        return response()->json($score);
    }

    public function destroy(Score $score)
    {
        $score->delete();
        return response()->json(['message' => 'Deleted']);
    }

    public function export(Request $request)
    {
        $query = Score::query()->orderByDesc('id');
        if ($request->filled('id_pelajaran')) {
            $query->where('id_pelajaran', (int)$request->input('id_pelajaran'));
        }
        if ($request->filled('tahun')) {
            $query->where('tahun', (string)$request->input('tahun'));
        }
        if ($request->filled('nisn')) {
            $query->where('nisn', (string)$request->input('nisn'));
        }
        $rows = $query->get();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $headers = [
            'ID','NISN','Nama','ID Pelajaran','Nama Pelajaran','STS Ganjil','SAS','STS Genap','SAT','US','Harian','Keterampilan','Keterangan','Guru','Tahun','Created At','Updated At'
        ];
        $sheet->fromArray($headers, null, 'A1');

        $data = [];
        foreach ($rows as $r) {
            $data[] = [
                $r->id,
                $r->nisn,
                $r->nama,
                $r->id_pelajaran,
                $r->name_pelajaran,
                $r->sts_ganjil,
                $r->sas,
                $r->sts_genap,
                $r->sat,
                $r->us,
                $r->harian,
                $r->keterampilan,
                $r->keterangan,
                $r->teacher,
                $r->tahun,
                $r->created_at,
                $r->updated_at,
            ];
        }
        if (!empty($data)) {
            $sheet->fromArray($data, null, 'A2');
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'scores_export_'.date('Ymd_His').'.xlsx';

        $response = new StreamedResponse(function () use ($writer) {
            $writer->save('php://output');
        });
        $response->headers->set('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        $response->headers->set('Content-Disposition', 'attachment;filename="'.$filename.'"');
        $response->headers->set('Cache-Control', 'max-age=0');

        return $response;
    }
}