<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\School;
use Illuminate\Support\Facades\Storage;

class SchoolController extends Controller
{
    public function index()
    {
        return response()->json(School::orderBy('nama')->get());
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nama' => 'required|string|max:255',
            'alamat' => 'nullable|string|max:255',
            'kepsek' => 'nullable|string|max:255',
            'kode' => 'nullable|string|max:100',
            'kota' => 'nullable|string|max:100',
            'provinsi' => 'nullable|string|max:100',
            'no' => 'nullable|string|max:100',
            'status' => 'nullable|string|in:Aktif,Tidak Aktif',
            'note' => 'nullable|string',
            'logo' => 'nullable|image|mimes:png,jpg,jpeg|max:2048',
            'signature' => 'nullable|image|mimes:png,jpg,jpeg|max:2048',
            'stamp' => 'nullable|image|mimes:png,jpg,jpeg|max:2048',
            // dukung path langsung ke folder public/images/school
            'logo_path' => 'nullable|string|max:255',
            'signature_path' => 'nullable|string|max:255',
            'stamp_path' => 'nullable|string|max:255',
        ]);

        $school = School::create([
            'nama' => $validated['nama'],
            'alamat' => $validated['alamat'] ?? null,
            'kepsek' => $validated['kepsek'] ?? null,
            'kode' => $validated['kode'] ?? null,
            'kota' => $validated['kota'] ?? null,
            'provinsi' => $validated['provinsi'] ?? null,
            'no' => $validated['no'] ?? null,
            'status' => $validated['status'] ?? 'Aktif',
            'note' => $validated['note'] ?? null,
        ]);

        // Simpan file assets jika ada
        $baseDir = 'schools/' . $school->id;
        Storage::disk('public')->makeDirectory($baseDir);
        $updates = [];
        foreach (['logo' => 'logo', 'signature' => 'signature', 'stamp' => 'stamp'] as $field => $name) {
            if ($request->hasFile($field)) {
                $ext = $request->file($field)->extension();
                $filename = $name . '.' . $ext;
                $path = $baseDir . '/' . $filename;
                Storage::disk('public')->put($path, file_get_contents($request->file($field)->getRealPath()));
                $updates[$name . '_path'] = $path;
            }
        }
        // terima path langsung jika dikirim oleh frontend
        foreach (['logo_path','signature_path','stamp_path'] as $col) {
            if ($request->filled($col)) {
                $updates[$col] = ltrim($request->input($col), '/');
            }
        }
        if (!empty($updates)) { $school->update($updates); }

        return response()->json($school->fresh(), 201);
    }

    public function show(School $school)
    {
        return response()->json($school);
    }

    public function update(Request $request, School $school)
    {
        $validated = $request->validate([
            'nama' => 'required|string|max:255',
            'alamat' => 'nullable|string|max:255',
            'kepsek' => 'nullable|string|max:255',
            'kode' => 'nullable|string|max:100',
            'kota' => 'nullable|string|max:100',
            'provinsi' => 'nullable|string|max:100',
            'no' => 'nullable|string|max:100',
            'status' => 'nullable|string|in:Aktif,Tidak Aktif',
            'note' => 'nullable|string',
            'logo' => 'nullable|image|mimes:png,jpg,jpeg|max:2048',
            'signature' => 'nullable|image|mimes:png,jpg,jpeg|max:2048',
            'stamp' => 'nullable|image|mimes:png,jpg,jpeg|max:2048',
            // dukung path langsung ke folder public/images/school
            'logo_path' => 'nullable|string|max:255',
            'signature_path' => 'nullable|string|max:255',
            'stamp_path' => 'nullable|string|max:255',
        ]);

        $school->update([
            'nama' => $validated['nama'],
            'alamat' => $validated['alamat'] ?? null,
            'kepsek' => $validated['kepsek'] ?? null,
            'kode' => $validated['kode'] ?? $school->kode,
            'kota' => $validated['kota'] ?? $school->kota,
            'provinsi' => $validated['provinsi'] ?? $school->provinsi,
            'no' => $validated['no'] ?? $school->no,
            'status' => $validated['status'] ?? $school->status,
            'note' => $validated['note'] ?? $school->note,
        ]);

        // Simpan file assets jika ada, ganti dan hapus lama
        $baseDir = 'schools/' . $school->id;
        Storage::disk('public')->makeDirectory($baseDir);
        $updates = [];
        $map = ['logo' => 'logo_path', 'signature' => 'signature_path', 'stamp' => 'stamp_path'];
        foreach ($map as $field => $col) {
            if ($request->hasFile($field)) {
                if (!empty($school->{$col})) {
                    Storage::disk('public')->delete($school->{$col});
                }
                $ext = $request->file($field)->extension();
                $filename = str_replace('_path','',$col) . '.' . $ext;
                $path = $baseDir . '/' . $filename;
                Storage::disk('public')->put($path, file_get_contents($request->file($field)->getRealPath()));
                $updates[$col] = $path;
            }
        }
        // terima path langsung jika dikirim oleh frontend
        foreach (['logo_path','signature_path','stamp_path'] as $col) {
            if ($request->filled($col)) {
                $updates[$col] = ltrim($request->input($col), '/');
            }
        }
        if (!empty($updates)) { $school->update($updates); }

        return response()->json($school->fresh());
    }

    public function destroy(School $school)
    {
        $school->delete();
        return response()->json(['message' => 'Deleted']);
    }
}