<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ExamParticipant;
use App\Models\ExamSession;
use App\Models\ExamResult;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Dompdf\Dompdf;
use Dompdf\Options;
use Carbon\Carbon;
 

class ExamParticipantController extends Controller
{
    private function generateExamPassword(int $digits = 8): string
    {
        $min = (int) pow(10, $digits - 1);
        $max = (int) pow(10, $digits) - 1;
        $tries = 0;
        do {
            $candidate = (string) random_int($min, $max);
            $exists = ExamParticipant::where('exam_password', $candidate)->exists();
            $tries++;
            if (!$exists) return $candidate;
        } while ($tries < 100);
        // Fallback jika kombinasi penuh: gunakan string acak 10 digit
        return (string) random_int(1000000000, 9999999999);
    }
    private function generateNumericToken(int $digits = 5): string
    {
        $min = (int) pow(10, $digits - 1);
        $max = (int) pow(10, $digits) - 1;
        $tries = 0;
        do {
            $candidate = (string) random_int($min, $max);
            $exists = ExamParticipant::where('login_token', $candidate)->exists();
            $tries++;
            if (!$exists) return $candidate;
        } while ($tries < 100);
        // Fallback bila penuh kombinasi: gunakan token acak panjang
        return Str::random(12);
    }
    public function index(Request $request)
    {
        $query = ExamParticipant::query()->orderByDesc('id');
        if ($request->filled('id_kelas')) {
            $query->where('id_kelas', (int)$request->input('id_kelas'));
        }
        // Jangan expose field sensitif (login_token, bound_device_hash)
        return response()->json($query->get()->makeHidden(['login_token','bound_device_hash']));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nisn' => 'required|string|unique:exam_participants,nisn',
            'nama' => 'required|string|max:255',
            'id_kelas' => 'required|exists:classes,id',
            'name_class' => 'required|string|max:255',
            'jurusan' => 'nullable|string|max:255',
            'password' => 'nullable|string|min:6',
        ]);

        $data = $validated;
        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->input('password'));
        } else {
            $data['password'] = Hash::make('12345678');
        }

        // Generate token login unik 5 digit saat pembuatan peserta
        $data['login_token'] = $this->generateNumericToken(5);

        $participant = ExamParticipant::create($data);
        return response()->json($participant, 201);
    }

    public function show(Request $request, ExamParticipant $examParticipant)
    {
        // Batasi akses: peserta ujian tidak boleh melihat detail peserta (termasuk token)
        $user = $request->user();
        if ($user instanceof \App\Models\ExamParticipant) {
            return response()->json(['message' => 'Hanya admin dapat melihat detail peserta'], 403);
        }
        // Tampilkan login_token dan exam_password untuk admin, tetap sembunyikan bound_device_hash
        return response()->json($examParticipant->makeVisible(['login_token', 'exam_password']));
    }

    public function update(Request $request, ExamParticipant $examParticipant)
    {
        $validated = $request->validate([
            'nisn' => 'required|string|unique:exam_participants,nisn,' . $examParticipant->id,
            'nama' => 'required|string|max:255',
            'id_kelas' => 'required|exists:classes,id',
            'name_class' => 'required|string|max:255',
            'jurusan' => 'nullable|string|max:255',
            'password' => 'nullable|string|min:6',
        ]);
        $data = $validated;
        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->input('password'));
        } else {
            unset($data['password']);
        }

        $examParticipant->update($data);
        return response()->json($examParticipant);
    }

    public function destroy(ExamParticipant $examParticipant)
    {
        $examParticipant->delete();
        return response()->json(['message' => 'Deleted']);
    }

    public function resetPassword(ExamParticipant $examParticipant)
    {
        $examParticipant->update(['password' => Hash::make('12345678')]);
        return response()->json(['message' => 'Password direset ke 12345678']);
    }

    /**
     * Hapus semua peserta ujian (opsional berdasarkan kelas) beserta sesi & hasilnya
     */
    public function destroyAll(Request $request)
    {
        $classId = $request->input('id_kelas');

        return DB::transaction(function () use ($classId) {
            $participantsQuery = ExamParticipant::query();
            if (!empty($classId)) {
                $participantsQuery->where('id_kelas', (int)$classId);
            }

            $participantIds = $participantsQuery->pluck('id')->all();
            if (empty($participantIds)) {
                return response()->json(['message' => 'Tidak ada peserta untuk dihapus', 'deleted' => ['participants' => 0, 'sessions' => 0, 'results' => 0]]);
            }

            // Ambil semua sesi yang terkait peserta-peserta ini
            $sessionIds = ExamSession::whereIn('exam_participant_id', $participantIds)->pluck('id')->all();

            // Hapus hasil ujian untuk sesi-sesi tersebut
            $deletedResults = 0;
            if (!empty($sessionIds)) {
                $deletedResults = ExamResult::whereIn('exam_session_id', $sessionIds)->delete();
            }

            // Hapus sesi ujian
            $deletedSessions = 0;
            if (!empty($sessionIds)) {
                $deletedSessions = ExamSession::whereIn('id', $sessionIds)->delete();
            }

            // Hapus peserta
            $deletedParticipants = ExamParticipant::whereIn('id', $participantIds)->delete();

            return response()->json([
                'message' => 'Peserta, sesi, dan hasil dihapus',
                'deleted' => [
                    'participants' => $deletedParticipants,
                    'sessions' => $deletedSessions,
                    'results' => $deletedResults,
                ],
            ]);
        });
    }

    // Reset binding perangkat: kosongkan fingerprint terikat sehingga peserta bisa login dari perangkat baru
    public function unbindDevice(Request $request, ExamParticipant $examParticipant)
    {
        // Batasi agar hanya admin (User dengan permission) yang boleh reset perangkat
        $actor = $request->user();
        if ($actor instanceof \App\Models\ExamParticipant) {
            return response()->json(['message' => 'Hanya admin dapat mereset perangkat peserta'], 403);
        }
        $examParticipant->update([
            'bound_device_hash' => null,
            'device_bound_at' => null,
        ]);
        return response()->json(['message' => 'Perangkat peserta berhasil direset']);
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv,txt|max:5120',
            'id_kelas' => 'nullable|exists:classes,id',
        ]);

        $idKelasDefault = $request->input('id_kelas');
        $nameClassDefault = null;
        if ($idKelasDefault) {
            $class = \App\Models\SchoolClass::find($idKelasDefault);
            if (!$class) return response()->json(['message' => 'Kelas tidak valid'], 422);
            $nameClassDefault = $class->name;
        }

        $file = $request->file('file');
        $path = $file->getRealPath();
        $ext = strtolower($file->getClientOriginalExtension());

        $rows = [];
        if (in_array($ext, ['csv', 'txt'])) {
            $handle = fopen($path, 'r');
            $header = null;
            while (($data = fgetcsv($handle, 0, ',')) !== false) {
                if (!$header) { $header = array_map(fn($v) => strtolower(trim((string)$v)), $data); continue; }
                $row = [];
                foreach ($header as $i => $col) { $row[$col] = $data[$i] ?? null; }
                $rows[] = $row;
            }
            fclose($handle);
        } else {
            $spreadsheet = IOFactory::load($path);
            $sheet = $spreadsheet->getActiveSheet()->toArray();
            $header = array_map(fn($v) => strtolower(trim((string) $v)), $sheet[0] ?? []);
            for ($i = 1; $i < count($sheet); $i++) {
                $line = $sheet[$i];
                if (count(array_filter($line, fn($v) => $v !== null && $v !== '')) === 0) continue;
                $row = [];
                foreach ($header as $idx => $col) { $row[$col] = $line[$idx] ?? null; }
                $rows[] = $row;
            }
        }

        $imported = 0; $updated = 0; $errors = [];
        foreach ($rows as $row) {
            $nisn = trim((string)($row['nisn'] ?? ''));
            $nama = trim((string)($row['nama'] ?? ''));
            $jurusan = isset($row['jurusan']) ? trim((string)$row['jurusan']) : null;
            // Password dari file (opsional). Jika kosong, akan dibuat otomatis 8 digit unik.
            $passwordPlain = isset($row['password']) ? trim((string)$row['password']) : '';
            // Normalisasi alias kolom untuk kompatibilitas variasi header
            if (!isset($row['id_kelas']) && isset($row['id kelas'])) {
                $row['id_kelas'] = $row['id kelas'];
            }
            // Dukung dua metode penentuan kelas:
            // 1) Default dari parameter id_kelas (dipilih di UI)
            // 2) Per baris: menggunakan kolom id_kelas atau kolom nama kelas ('kelas')
            $id_kelas = $idKelasDefault ?: (isset($row['id_kelas']) ? (int)$row['id_kelas'] : null);
            $kelasByName = '';
            if (isset($row['kelas'])) {
                $kelasByName = trim((string)$row['kelas']);
            } elseif (isset($row['class'])) {
                $kelasByName = trim((string)$row['class']);
            }

            $kelasModel = null;
            if (!$id_kelas && $kelasByName !== '') {
                // Cocokkan nama kelas secara case-insensitive, setelah menormalkan spasi berlebih
                $normalizedName = preg_replace('/\s+/', ' ', strtolower($kelasByName));
                $kelasModel = \App\Models\SchoolClass::whereRaw('LOWER(REPLACE(name, "  ", " ")) = ?', [$normalizedName])->first();
                if ($kelasModel) {
                    $id_kelas = (int)$kelasModel->id;
                } else {
                    $errors[] = ['row' => $row, 'message' => "kelas '$kelasByName' tidak ditemukan"];
                    continue;
                }
            }

            if (!$nisn || !$nama || !$id_kelas) {
                $errors[] = ['row' => $row, 'message' => 'Kolom wajib tidak lengkap (nisn, nama, id_kelas/kelas)'];
                continue;
            }

            // Tentukan name_class untuk penyimpanan
            $name_class = $nameClassDefault;
            if (!$name_class) {
                if ($kelasModel) {
                    $name_class = $kelasModel->name;
                } else {
                    $kelas = \App\Models\SchoolClass::find($id_kelas);
                    if (!$kelas) { $errors[] = ['row' => $row, 'message' => 'Kelas tidak ditemukan untuk id_kelas yang diberikan']; continue; }
                    $name_class = $kelas->name;
                }
            }

            $existing = ExamParticipant::where('nisn', $nisn)->first();
            if ($existing) {
                $updateData = [
                    'nama' => $nama,
                    'id_kelas' => $id_kelas,
                    'name_class' => $name_class,
                    'jurusan' => $jurusan,
                ];
                // Jika file menyediakan password, update keduanya (hash dan exam_password)
                if ($passwordPlain !== '') {
                    $updateData['password'] = Hash::make($passwordPlain);
                    $updateData['exam_password'] = $passwordPlain;
                }
                // Jika belum memiliki token, buatkan token 5 digit
                if (empty($existing->login_token)) {
                    $updateData['login_token'] = $this->generateNumericToken(5);
                }
                $existing->update($updateData);
                $updated++;
            } else {
                // Buat password acak 8 digit jika tidak diberikan
                $finalPlain = $passwordPlain !== '' ? $passwordPlain : $this->generateExamPassword(8);
                ExamParticipant::create([
                    'nisn' => $nisn,
                    'nama' => $nama,
                    'id_kelas' => $id_kelas,
                    'name_class' => $name_class,
                    'jurusan' => $jurusan,
                    'password' => Hash::make($finalPlain),
                    'exam_password' => $finalPlain,
                    'login_token' => $this->generateNumericToken(5),
                ]);
                $imported++;
            }
        }

        return response()->json(['imported' => $imported, 'updated' => $updated, 'errors' => $errors]);
    }

    // Export token peserta (nisn, nama, kelas, token) ke Excel
    public function exportTokens(Request $request)
    {
        $query = ExamParticipant::query()->orderBy('id');
        if ($request->filled('id_kelas')) {
            $query->where('id_kelas', (int)$request->input('id_kelas'));
        }
        $rows = $query->get(['nisn','nama','name_class','login_token']);

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $headers = ['nisn', 'nama', 'kelas', 'token'];
        foreach ($headers as $i => $h) {
            $colLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($i + 1);
            $sheet->setCellValue($colLetter . '1', $h);
            $sheet->getColumnDimension($colLetter)->setAutoSize(true);
        }
        $rowIdx = 2;
        foreach ($rows as $r) {
            $sheet->setCellValue('A'.$rowIdx, $r->nisn);
            $sheet->setCellValue('B'.$rowIdx, $r->nama);
            $sheet->setCellValue('C'.$rowIdx, $r->name_class);
            $sheet->setCellValue('D'.$rowIdx, $r->login_token ?? '');
            $rowIdx++;
        }

        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        ob_start();
        $writer->save('php://output');
        $excelOutput = ob_get_clean();

        $filename = 'tokens_peserta.xlsx';
        return response($excelOutput, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"',
        ]);
    }

    // Generate token massal untuk peserta yang belum memiliki token
    public function generateTokens(Request $request)
    {
        $validated = $request->validate([
            'id_kelas' => 'nullable|exists:classes,id',
        ]);

        $query = ExamParticipant::query();
        if (!empty($validated['id_kelas'])) {
            $query->where('id_kelas', $validated['id_kelas']);
        }
        $query->where(function($q){
            $q->whereNull('login_token')->orWhere('login_token', '');
        });

        $toGenerate = $query->get();
        $count = 0;
        foreach ($toGenerate as $p) {
            $p->login_token = $this->generateNumericToken(5);
            $p->save();
            $count++;
        }

        return response()->json(['generated' => $count]);
    }

    public function template(Request $request)
    {
        $idKelas = $request->query('id_kelas');
        if (empty($idKelas) || !\App\Models\SchoolClass::find((int)$idKelas)) {
            return response()->json(['message' => 'id_kelas wajib dipilih dan valid'], 422);
        }

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Password tidak perlu diisi di template karena akan dibuat otomatis 8 digit saat import.
        // Karena kelas sudah dipilih di UI, template hanya membutuhkan kolom nisn dan nama.
        $headers = ['nisn', 'nama'];
        foreach ($headers as $i => $h) {
            $colLetter = Coordinate::stringFromColumnIndex($i + 1);
            $sheet->setCellValue($colLetter . '1', $h);
            $sheet->getColumnDimension($colLetter)->setAutoSize(true);
        }

        $writer = new Xlsx($spreadsheet);
        ob_start();
        $writer->save('php://output');
        $excelOutput = ob_get_clean();

        return response($excelOutput, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="exam-participants.xlsx"',
        ]);
    }

    // Template import alternatif: nisn, nama, kelas (by name)
    public function templateByName(Request $request)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['nisn', 'nama', 'kelas'];
        foreach ($headers as $i => $h) {
            $colLetter = Coordinate::stringFromColumnIndex($i + 1);
            $sheet->setCellValue($colLetter . '1', $h);
            $sheet->getColumnDimension($colLetter)->setAutoSize(true);
        }

        // Opsional: tambahkan petunjuk singkat di baris ke-2
        $sheet->setCellValue('A2', '1234567890');
        $sheet->setCellValue('B2', 'Nama Peserta');
        $sheet->setCellValue('C2', 'XI IPA 1');

        $writer = new Xlsx($spreadsheet);
        ob_start();
        $writer->save('php://output');
        $excelOutput = ob_get_clean();

        return response($excelOutput, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="exam-participants-by-name.xlsx"',
        ]);
    }

    // Bulk reset password to default (12345678) for all or per class
    public function resetPasswordBulk(Request $request)
    {
        $validated = $request->validate([
            'id_kelas' => 'nullable|exists:classes,id',
        ]);

        $query = ExamParticipant::query();
        if (!empty($validated['id_kelas'])) {
            $query->where('id_kelas', $validated['id_kelas']);
        }

        $updated = $query->update(['password' => Hash::make('12345678')]);
        return response()->json(['updated' => $updated, 'message' => 'Password peserta berhasil direset']);
    }

    // Export PDF: gabungkan Nomor Meja (kiri) dan Kartu Ujian (kanan), 5 baris per halaman
    public function exportCardsSeats(Request $request)
{
    $validated = $request->validate([
        'id_kelas' => 'required|exists:classes,id',
        'rows_per_page' => 'nullable|integer|min:4|max:12',
        'scale' => 'nullable|numeric|min:0.5|max:1.0',
    ]);

    try {
        // Kurangi risiko kehabisan memori saat render PDF untuk kelas besar
        @ini_set('memory_limit', '512M');
        @set_time_limit(120);

        $class = \App\Models\SchoolClass::with(['school','grade'])->find((int)$validated['id_kelas']);
        if (!$class) {
            return response()->json(['message' => 'Kelas tidak ditemukan'], 404);
        }

        $school = $class->school;
        $grade = $class->grade;
        $participants = ExamParticipant::where('id_kelas', $class->id)
            ->orderBy('nama', 'asc')
            ->get(['id','nisn','nama','id_kelas','name_class','login_token','exam_password']);

        if ($participants->isEmpty()) {
            return response()->json(['message' => 'Tidak ada peserta pada kelas ini'], 422);
        }

        $schoolName = htmlspecialchars((string)($school?->nama ?? ''));
        $className = htmlspecialchars((string)$class->name);
        $gradeName = htmlspecialchars((string)($grade?->grade ?? ''));
        // Password pada kartu diambil dari exam_password yang dibuat saat import
        $address = htmlspecialchars((string)($school?->alamat ?? ''));
        $dateStr = \Carbon\Carbon::now()->locale('id')->isoFormat('D MMMM YYYY');
        $yearNow = \Carbon\Carbon::now()->format('Y');
        $cityName = htmlspecialchars((string)($school?->kota ?? ''));
        $schoolCode = htmlspecialchars((string)($school?->kode ?? ''));

        // --- fungsi helper gambar ---
        $imgToDataUri = function (?string $path): ?string {
            if (!$path) return null;
            try {
                if (preg_match('#^https?://#', $path)) {
                    $content = @file_get_contents($path);
                } elseif (Storage::disk('public')->exists($path)) {
                    $content = Storage::disk('public')->get($path);
                } elseif (file_exists(public_path($path))) {
                    $content = file_get_contents(public_path($path));
                } else {
                    return null;
                }
                if (!$content) return null;
                $ext = pathinfo($path, PATHINFO_EXTENSION) ?: 'png';
                $mime = 'image/' . ($ext === 'jpg' ? 'jpeg' : $ext);
                return 'data:' . $mime . ';base64,' . base64_encode($content);
            } catch (\Throwable $e) {
                return null;
            }
        };

        $logoSrc = $imgToDataUri((string)($school?->logo_path));
        $signSrc = $imgToDataUri((string)($school?->signature_path));
        $stampSrc = $imgToDataUri((string)($school?->stamp_path));
        $eduLogoSrc = $imgToDataUri('data/pendidikan.png');
        // Tidak gunakan background-image agar gambar tajam; CSS tambahan minimal
        $imageCss = '';

        // ========================
        // SUSUN HTML
        // ========================
        $rowsHtml = '';
        $rowsPerPage = (int)($validated['rows_per_page'] ?? 4);
        $rowsPerPage = max(4, min(12, $rowsPerPage));
        $scale = (float)($validated['scale'] ?? 1.0);
        $scale = max(0.5, min(1.0, $scale));
        $idx = 0;
        foreach ($participants as $p) {
            $idx++;
            $seatNo = $idx;
            $nama = htmlspecialchars((string)$p->nama);
            $nisn = htmlspecialchars((string)$p->nisn);
            $token = htmlspecialchars((string)($p->login_token ?? ''));
            $participantNo = htmlspecialchars((string)($yearNow . '-' . $gradeName . '-' . $p->id));
            $examPasswordRaw = $p->exam_password ?? '';
            if ($examPasswordRaw === '') { $examPasswordRaw = '12345678'; }
            $examPassword = htmlspecialchars((string)$examPasswordRaw);

            $rowsHtml .= "
<table class='row-block' style='width:100%; border-collapse:separate; border-spacing:0 2mm; margin-bottom:0; table-layout:fixed;'>
<tr>
<td style='width:50%; vertical-align:top; padding:0; height:auto; text-align:left;'>
    <div class=\"card-box\" style='display:inline-block; width:95mm; height:60mm; border:1px solid #000; border-radius:1.5mm; padding:2mm;'>
    <div class=\"card-content\" style='height:100%;'>
        <div class=\"kop-wrap\">
            ".($logoSrc ? "<img class=\"logo-left\" src=\"$logoSrc\" alt=\"logo sekolah\"/>" : "")."
            ".($eduLogoSrc ? "<img class=\"logo-right\" src=\"$eduLogoSrc\" alt=\"logo pendidikan\"/>" : "")."
            <div class=\"kop-center\">
                <div class=\"kop-title\" style='font-size:10pt; font-weight:bold;'>NOMOR MEJA</div>
                <div class=\"kop-school\" style='font-size:8pt;'>$schoolName</div>
                <div class=\"kop-address\" style='font-size:6pt;'>$address</div>
            </div>
        </div>
        <div class=\"line\" style='border-top:1px solid #000; margin:2px 0;'></div>
        <div style='text-align:center; margin-top:2mm;'>
            <div style='font-weight:bold; font-size:9pt;'>Nomor Meja</div>
            <div style='font-size:30pt; line-height:1;'>$seatNo</div>
            <div style='font-weight:bold; font-size:12pt;'>$participantNo</div>
            <div style='font-weight:bold; font-size:12pt;'>$nama</div>

        </div>
    </div>
    </div>
</td>

<td style='width:50%; vertical-align:top; padding:0; height:auto; text-align:right;'>
    <div class=\"card-box\" style='display:inline-block; width:95mm; height:60mm; border:1px solid #000; border-radius:1.5mm; padding:2mm;'>
    <div class=\"card-content\" style='height:100%;'>
        <!-- ✅ KOP DIPINDAH KE ATAS -->
        <div class=\"kop-wrap\">
            ".($logoSrc ? "<img class=\"logo-left\" src=\"$logoSrc\" alt=\"logo sekolah\"/>" : "")."
            ".($eduLogoSrc ? "<img class=\"logo-right\" src=\"$eduLogoSrc\" alt=\"logo pendidikan\"/>" : "")."
            <div class=\"kop-center\">
                <div class=\"kop-title\" style='font-size:10pt; font-weight:bold;'>KARTU PESERTA</div>
                <div class=\"kop-school\" style='font-size:8pt;'>$schoolName</div>
                <div class=\"kop-address\" style='font-size:6pt;'>$address</div>
            </div>
        </div>

        <div class=\"line\" style='border-top:1px solid #000; margin:2px 0;'></div>
        <div class='content-columns'>
        <div class='details-wrap'>
            <table style='width:100%; font-size:7pt; border-collapse:collapse;'>
                <tr><td style='width:90px;'>No Peserta</td><td>:&#9;$participantNo</td></tr>
                <tr><td>Nama</td><td>:&#9;$nama</td></tr>
                <tr><td>NISN</td><td>:&#9;$nisn</td></tr>
                <tr><td>Jenjang</td><td>:&#9;$gradeName</td></tr>
            </table>
            <div class='details-bottom'>
                <table style='width:100%; font-size:7pt; border-collapse:collapse;'>
                    <tr><td style='width:90px;'>Kelas</td><td>:&#9;$className</td></tr>
                    <tr><td>Password</td><td>:&#9;$examPassword</td></tr>
                    <tr><td>Token</td><td>:&#9;$token</td></tr>
                </table>
            </div>
        </div>
        <div class=\"signature-col\"> 
        <div class=\"signature\" style='margin-top:0mm; font-size:6pt;'>
            <div class=\"addr\">$cityName, $dateStr</div>
            ".($stampSrc ? "<img class=\"stamp\" src=\"$stampSrc\" alt=\"cap\"/>" : "")."
            ".($signSrc ? "<img class=\"sign\" src=\"$signSrc\" alt=\"ttd\"/>" : "")."
            <div class=\"kepsek-info\">
                <div class=\"kepsek-name\">" . htmlspecialchars((string)($school?->kepsek ?? '')) . "</div>
            </div>
        </div>
        </div>

        
    </div>
    </div>
</td>
</tr>
</table>
";

            // Biarkan halaman berganti otomatis saat konten mencapai batas kertas
        }

        $html = "<!DOCTYPE html><html><head><meta charset='utf-8'>
        <style>
        @page { size: 215mm 330mm; margin: 6mm; }
        body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 10pt; color: #111; }
        table { width: 100%; border-collapse: collapse; }
        .page-break { page-break-after: always; }
        .page-container { width: 203mm; margin: 0 auto; padding: 0 2mm; box-sizing: border-box; }
        .scale-wrap { transform: scale($scale); transform-origin: top center; display:block; }
        .card-box { box-sizing: border-box; }
        table.row-block { page-break-inside: avoid; table-layout: fixed; }
        .kop-wrap { position: relative; height: 21mm; overflow: hidden; }
        .logo-left, .logo-right { position: absolute; top: 3mm; height: 16mm; width: auto; opacity: 1; z-index: 2; image-rendering: auto; }
        .logo-left { left: 2mm; }
        .logo-right { right: 2mm; }
        .kop-center { position: absolute; left: 0; right: 0; top: 5mm; text-align: center; z-index: 3; }
        .kop-title { font-weight: 700; font-size: 10pt; }
        .kop-school { font-size: 9pt; }
        .kop-address { font-size: 7pt; color: #444; }
        .content-columns { display: flex; gap: 2mm; align-items: flex-start; justify-content: flex-start; }
        .signature-col { width: 30mm; display: flex; flex-direction: column; align-items: flex-end; margin-left: auto; position: relative; z-index: 10; }
        .signature { position: relative; min-height: 26mm; overflow: visible; width: 30mm; margin-top: -4mm; }
        .signature .addr { position: absolute; left: 0; right: 0; top: 0; text-align: left; font-size: 4pt; line-height: 1; z-index: 9; white-space: nowrap; overflow: visible; padding: 0 1mm; }
        .signature .stamp { position: absolute; right: 6mm; bottom: 0mm; width: 18mm; height: auto; opacity: 1.0; z-index: 4; }
        .signature .sign { position: absolute; right: 6mm; bottom: 0mm; width: 18mm; height: auto; opacity: 1.0; z-index: 5; }
        .kepsek-info { position: absolute; left: 1mm; right: 6mm; top: 8mm; bottom: auto; text-align: left; font-size: 4pt; line-height: 1.1; z-index: 8; white-space: nowrap; overflow: visible; }
        $imageCss
        .kepsek-info .kepsek-name { font-weight: bold;  display:inline-block; max-width: none; white-space: nowrap; overflow: visible; border-bottom:1px solid #000; padding-bottom:1px; }
        .details-wrap { width: 58mm; margin-right: 0; float: none; }
        .details-wrap table { table-layout: fixed; }
        .details-wrap td { white-space: nowrap; }
        .details-bottom { margin-top: 1mm; }
        </style></head><body><div class='page-container'><div class='scale-wrap'>$rowsHtml</div></div></body></html>";

        // render PDF
        $dompdf = new \Dompdf\Dompdf();
        // Kurangi DPI agar penggunaan memori lebih hemat & izinkan remote bila diperlukan
        $options = $dompdf->getOptions();
        $options->set('dpi', 300);
        $options->set('isRemoteEnabled', true);
        $dompdf->setOptions($options);
        $dompdf->loadHtml($html);
        // F4 in points ≈ 610 x 935 pt at 72 DPI
        $dompdf->setPaper([0, 0, 610, 935], 'portrait');
        $dompdf->render();
        $filename = 'kartu_ujian_nomor_meja_kelas_'.$class->id.'.pdf';
        return response($dompdf->output(), 200, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"'
        ]);
    } catch (\Throwable $e) {
        return response()->json(['message' => $e->getMessage()], 500);
    }
}


    // Update activity ping for exam participants (by id or nisn)
    public function updateActivity(Request $request)
    {
        $request->validate([
            'id' => 'nullable|exists:exam_participants,id',
            'nisn' => 'nullable|string',
        ]);

        $participant = null;
        if ($request->filled('id')) {
            $participant = ExamParticipant::find($request->input('id'));
        } elseif ($request->filled('nisn')) {
            $participant = ExamParticipant::where('nisn', $request->input('nisn'))->first();
        }

        if (!$participant) {
            return response()->json(['message' => 'Peserta tidak ditemukan'], 404);
        }

        $participant->last_activity = now();
        $participant->save();

        return response()->json(['message' => 'Aktivitas peserta diperbarui']);
    }

    // [REMOVED] exportTokensJson: endpoint khusus load test dihapus sesuai permintaan
}