<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Exam;
use App\Models\ExamSession;
use App\Models\Question;
use App\Models\ExamResult;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Carbon\Carbon;
use App\Jobs\ProcessFinalExamSubmission;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class ExamController extends Controller
{
    /**
     * Normalisasi teks untuk perbandingan: huruf kecil, transliterasi, hapus tanda baca, rapikan spasi
     */
    private function normalizeText(string $text): string
    {
        $t = trim($text);
        // Transliterasi untuk menghapus diakritik bila memungkinkan
        $t = @iconv('UTF-8', 'ASCII//TRANSLIT', $t) ?: $t;
        $t = mb_strtolower($t, 'UTF-8');
        // Hapus tanda baca umum
        $t = preg_replace('/[\p{P}\p{S}]+/u', ' ', $t);
        // Rapikan spasi
        $t = preg_replace('/\s+/u', ' ', $t);
        return trim((string)$t);
    }

    /**
     * Pecah teks menjadi himpunan kata unik, singkirkan stopwords umum bahasa Indonesia
     */
    private function wordSet(string $text): array
    {
        $normalized = $this->normalizeText($text);
        $parts = preg_split('/\s+/u', $normalized, -1, PREG_SPLIT_NO_EMPTY) ?: [];
        $stop = [
            'dan','yang','di','ke','dari','untuk','dengan','atau','pada','sebagai','adalah','itu','ini','kami','kita','saya','aku','anda','para','serta','hingga','agar','karena','namun','bahwa','jadi','dalam','akan','atau','tidak','ya','bukan','oleh','sebuah','suatu','sebuah','terhadap','lebih','kurang','lain','juga'
        ];
        $stopSet = array_fill_keys($stop, true);
        $tokens = [];
        foreach ($parts as $p) {
            if (mb_strlen($p, 'UTF-8') < 2) continue; // singkirkan kata sangat pendek
            if (isset($stopSet[$p])) continue;
            $tokens[$p] = true;
        }
        return array_keys($tokens);
    }

    /**
     * Hitung kemiripan berbasis karakter (similar_text %) dan Jaccard berbasis kata
     */
    private function similarityMetrics(string $a, string $b): array
    {
        $an = $this->normalizeText($a);
        $bn = $this->normalizeText($b);

        $percent = 0.0;
        similar_text($an, $bn, $percent);

        $wa = $this->wordSet($a);
        $wb = $this->wordSet($b);
        $setA = array_fill_keys($wa, true);
        $setB = array_fill_keys($wb, true);
        $inter = 0; $union = 0;
        $seen = [];
        foreach ($setA as $k => $_) { $seen[$k] = true; if (isset($setB[$k])) $inter++; $union++; }
        foreach ($setB as $k => $_) { if (!isset($seen[$k])) $union++; }
        $jaccard = $union > 0 ? ($inter / $union) : 0.0;

        return [
            'char_percent' => $percent,
            'jaccard' => $jaccard,
        ];
    }
    public function index(Request $request)
    {
        $query = Exam::query()->orderByDesc('id');
        if ($request->filled('id_subject')) {
            $query->where('id_subject', (int)$request->input('id_subject'));
        }
        return response()->json($query->get());
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'id_subject' => ['required', 'integer', 'exists:subjects,id'],
            'duration_minutes' => ['required', 'integer', 'min:1'],
            'scheduled_at' => ['required', 'date'],
            'id_school' => ['nullable', 'integer', 'exists:schools,id'],
            'id_grade' => ['nullable', 'integer', 'exists:grades,id'],
        ]);

        // Generate unique 6 uppercase exam code
        $code = null;
        for ($i = 0; $i < 5; $i++) {
            $candidate = Str::upper(Str::random(6));
            if (!Exam::where('code', $candidate)->exists()) {
                $code = $candidate;
                break;
            }
        }
        if (!$code) { $code = Str::upper(Str::random(6)); }

        $scheduledAt = Carbon::parse($validated['scheduled_at'], config('app.timezone'));

        $exam = Exam::create([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'id_subject' => $validated['id_subject'],
            'code' => $code,
            'duration_minutes' => $validated['duration_minutes'],
            'scheduled_at' => $scheduledAt,
            'id_school' => $request->input('id_school'),
            'id_grade' => $request->input('id_grade'),
        ]);
        return response()->json($exam, 201);
    }

    public function show(Exam $exam)
    {
        return response()->json($exam);
    }

    public function update(Request $request, Exam $exam)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'id_subject' => ['required', 'integer', 'exists:subjects,id'],
            'duration_minutes' => ['required', 'integer', 'min:1'],
            'scheduled_at' => ['required', 'date'],
            'id_school' => ['nullable', 'integer', 'exists:schools,id'],
            'id_grade' => ['nullable', 'integer', 'exists:grades,id'],
        ]);

        $scheduledAt = Carbon::parse($validated['scheduled_at'], config('app.timezone'));

        $exam->update([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'id_subject' => $validated['id_subject'],
            'duration_minutes' => $validated['duration_minutes'],
            'scheduled_at' => $scheduledAt,
            'id_school' => $request->input('id_school'),
            'id_grade' => $request->input('id_grade'),
        ]);
        return response()->json($exam);
    }

    public function destroy(Exam $exam)
    {
        $exam->delete();
        return response()->json(['message' => 'Deleted']);
    }

    /**
     * Hapus semua ujian (opsional: dibatasi oleh id_subject) beserta sesi dan hasilnya.
     * Endpoint ini untuk superadmin.
     */
    public function destroyAll(Request $request)
    {
        $subjectId = $request->input('id_subject');

        return DB::transaction(function () use ($subjectId) {
            $examQuery = Exam::query();
            if (!empty($subjectId)) {
                $examQuery->where('id_subject', (int)$subjectId);
            }

            $examIds = $examQuery->pluck('id')->all();
            if (empty($examIds)) {
                return response()->json([
                    'deleted_exams' => 0,
                    'deleted_sessions' => 0,
                    'deleted_results' => 0,
                    'message' => 'Tidak ada ujian yang cocok untuk dihapus'
                ]);
            }

            $sessionIds = ExamSession::whereIn('exam_id', $examIds)->pluck('id')->all();

            $deletedResults = 0;
            if (!empty($sessionIds)) {
                $deletedResults = ExamResult::whereIn('exam_session_id', $sessionIds)->delete();
            }

            $deletedSessions = 0;
            if (!empty($sessionIds)) {
                $deletedSessions = ExamSession::whereIn('id', $sessionIds)->delete();
            }

            $deletedExams = Exam::whereIn('id', $examIds)->delete();

            return response()->json([
                'deleted_exams' => $deletedExams,
                'deleted_sessions' => $deletedSessions,
                'deleted_results' => $deletedResults,
                'message' => 'Berhasil menghapus ujian beserta sesi dan hasilnya'
            ]);
        });
    }

    // List ujian yang dijadwalkan hari ini, disesuaikan dengan sekolah & grade peserta
    public function todayForParticipant(Request $request)
    {
        $today = now()->toDateString();

        // Karena route ini publik, pastikan tetap mendeteksi user peserta
        // saat ada Bearer token tanpa middleware auth:sanctum.
        // Gunakan fallback ke guard sanctum.
        $user = $request->user() ?? auth('sanctum')->user();

        // Ambil semua ujian hari ini bila bukan peserta (fallback admin)
        if (!($user instanceof \App\Models\ExamParticipant)) {
            $exams = Exam::with('subject')
                ->whereDate('scheduled_at', $today)
                ->orderBy('scheduled_at')
                ->get();
        } else {
            // Filter berdasarkan sekolah & grade peserta, prioritaskan cohort di level Exam
            $class = \App\Models\SchoolClass::find($user->id_kelas);
            $schoolId = $class?->id_school;
            $gradeId = $class?->id_grade;

            $exams = Exam::query()->with('subject')
                ->whereDate('scheduled_at', $today)
                ->where(function ($q) use ($schoolId, $gradeId) {
                    // Branch A: Exam memiliki id_school/id_grade spesifik dan harus cocok dengan peserta
                    $q->where(function ($qe) use ($schoolId, $gradeId) {
                        if ($schoolId) { $qe->where('id_school', $schoolId); } else { $qe->whereNull('id_school'); }
                        if ($gradeId) { $qe->where('id_grade', $gradeId); } else { $qe->whereNull('id_grade'); }
                    })
                    // Branch B: Exam tidak menentukan cohort, fallback ke Subject (atau tidak ada subject)
                    ->orWhere(function ($qf) use ($schoolId, $gradeId) {
                        $qf->whereNull('id_school')->whereNull('id_grade')
                            ->where(function ($qsOuter) use ($schoolId, $gradeId) {
                                $qsOuter->whereHas('subject', function ($qs) use ($schoolId, $gradeId) {
                                    if ($schoolId) {
                                        $qs->where(function ($qq) use ($schoolId) {
                                            $qq->where('id_school', $schoolId)->orWhereNull('id_school');
                                        });
                                    }
                                    if ($gradeId) {
                                        $qs->where(function ($qq) use ($gradeId) {
                                            $qq->where('id_grade', $gradeId)->orWhereNull('id_grade');
                                        });
                                    }
                                })
                                ->orDoesntHave('subject');
                            });
                    });
                })
                ->orderBy('scheduled_at')
                ->get();
        }



        // Tambahkan metadata waktu, status bisa mulai, ringkasan sesi, dan status sesi milik peserta saat ini
        $data = $exams->map(function ($exam) use ($user) {
            $start = \Carbon\Carbon::parse($exam->scheduled_at, config('app.timezone'));
            $end = (clone $start)->addMinutes((int)$exam->duration_minutes);
            $now = now();
            $canStart = $now->between($start, $end);

            // Hitung sesi ujian: sedang mengerjakan (started_at != null & finished_at == null)
            // dan sudah selesai (finished_at != null). Jika subject memiliki school/grade,
            // batasi ke peserta dengan kelas yang cocok.
            $sessionQueryBase = \App\Models\ExamSession::where('exam_id', $exam->id);
            // Prefer cohort Exam jika ada; jika tidak, gunakan Subject
            if ($exam->id_school || $exam->id_grade) {
                $sessionQueryBase = $sessionQueryBase->whereHas('participant', function ($qp) use ($exam) {
                    $qp->whereHas('kelas', function ($qk) use ($exam) {
                        if ($exam->id_school) { $qk->where('id_school', $exam->id_school); }
                        if ($exam->id_grade) { $qk->where('id_grade', $exam->id_grade); }
                    });
                });
            } elseif ($exam->subject && ($exam->subject->id_school || $exam->subject->id_grade)) {
                $sessionQueryBase = $sessionQueryBase->whereHas('participant', function ($qp) use ($exam) {
                    $qp->whereHas('kelas', function ($qk) use ($exam) {
                        if ($exam->subject->id_school) { $qk->where('id_school', $exam->subject->id_school); }
                        if ($exam->subject->id_grade) { $qk->where('id_grade', $exam->subject->id_grade); }
                    });
                });
            }

            $inProgressCount = (clone $sessionQueryBase)
                ->whereNotNull('started_at')
                ->whereNull('finished_at')
                ->count();
            $finishedCount = (clone $sessionQueryBase)
                ->whereNotNull('finished_at')
                ->count();

            // Perkiraan jumlah peserta yang belum mulai berdasarkan subjek sekolah/grade.
            // Bila subject memiliki id_school/id_grade, gunakan total peserta di kelas yang cocok.
            // Jika tidak, nilai ini tidak dapat ditentukan secara akurat (set ke null).
            $notStartedCount = null;
            if ($exam->id_school || $exam->id_grade) {
                $totalParticipants = \App\Models\ExamParticipant::query()
                    ->whereHas('kelas', function ($q) use ($exam) {
                        if ($exam->id_school) { $q->where('id_school', $exam->id_school); }
                        if ($exam->id_grade) { $q->where('id_grade', $exam->id_grade); }
                    })
                    ->count();
                $notStartedCount = max(0, $totalParticipants - ($inProgressCount + $finishedCount));
            } elseif ($exam->subject && ($exam->subject->id_school || $exam->subject->id_grade)) {
                $totalParticipants = \App\Models\ExamParticipant::query()
                    ->whereHas('kelas', function ($q) use ($exam) {
                        if ($exam->subject->id_school) { $q->where('id_school', $exam->subject->id_school); }
                        if ($exam->subject->id_grade) { $q->where('id_grade', $exam->subject->id_grade); }
                    })
                    ->count();
                $notStartedCount = max(0, $totalParticipants - ($inProgressCount + $finishedCount));
            }
            // Status sesi milik peserta saat ini (jika request dilakukan oleh peserta)
            $mySession = null;
            $myStatus = null;
            $myResult = null;
            if ($user instanceof \App\Models\ExamParticipant) {
                $session = \App\Models\ExamSession::where('exam_id', $exam->id)
                    ->where('exam_participant_id', $user->id)
                    ->first();
                if ($session) {
                    $attempts = \Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','submissions_count') ? (int)$session->submissions_count : 0;
                    $entryAttempts = \Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','entry_attempts') ? (int)$session->entry_attempts : null;
                    $finishRequested = \Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','finish_requested_at') && (bool)$session->finish_requested_at;
                    $hasResult = (bool) $session->result;
                    $expiredNow = now()->gt($end);
                    // Map ke status tampilan (selaras alur peserta)
                    if ($hasResult) {
                        $myStatus = 'success';
                    } elseif ($finishRequested) {
                        $myStatus = 'proses_hasil';
                    } elseif ($entryAttempts !== null && $entryAttempts > 3) {
                        // Lebih dari 3x masuk sesi: proses hasil
                        $myStatus = 'proses_hasil';
                    } elseif ($session->started_at) {
                        // 1–3x: sedang dikerjakan
                        $myStatus = 'mulai';
                    } else {
                        // belum mulai (punya row tapi belum start)
                        $myStatus = 'belum_mulai';
                    }
                    $mySession = [
                        'id' => $session->id,
                        'started_at' => optional($session->started_at)->toDateTimeString(),
                        'finished_at' => optional($session->finished_at)->toDateTimeString(),
                        'finish_requested_at' => optional($session->finish_requested_at)->toDateTimeString(),
                        'submissions_count' => $attempts,
                        'entry_attempts' => $entryAttempts,
                    ];
                    // Sertakan hasil (score) bila tersedia untuk sesi milik peserta
                    $res = $session->result;
                    if ($res) {
                        $myResult = [
                            'score' => $res->score,
                            'correct_count' => $res->correct_count,
                            'wrong_count' => $res->wrong_count,
                            'total_count' => $res->total_count,
                        ];
                    }
                } else {
                    // Tidak ada sesi: cek expired atau belum mulai
                    if (now()->gt($end)) {
                        $myStatus = 'expired';
                    } else {
                        $myStatus = 'belum_mulai';
                    }
                }
            }

            return [
                'id' => $exam->id,
                'name' => $exam->name,
                'description' => $exam->description,
                'code' => $exam->code,
                'id_subject' => $exam->id_subject,
                'subject_name' => $exam->subject?->name,
                'subject_code' => $exam->subject?->code,
                'duration_minutes' => $exam->duration_minutes,
                'scheduled_at' => $exam->scheduled_at,
                'start_at' => $start->toDateTimeString(),
                'end_at' => $end->toDateTimeString(),
                // ISO 8601 dengan offset timezone untuk kompatibilitas parsing di browser
                'start_at_iso' => $start->toIso8601String(),
                'end_at_iso' => $end->toIso8601String(),
                // Info subject untuk memastikan kecocokan school & grade
                'subject_school_id' => $exam->subject?->id_school,
                'subject_grade_id' => $exam->subject?->id_grade,
                // Tambahkan cohort di level exam bila tersedia
                'exam_school_id' => $exam->id_school,
                'exam_grade_id' => $exam->id_grade,
                'can_start' => $canStart,
                // Ringkasan status sesi untuk dashboard
                'in_progress_count' => $inProgressCount,
                'finished_count' => $finishedCount,
                'not_started_count' => $notStartedCount,
                // Status sesi milik peserta saat ini
                'my_status' => $myStatus,
                'my_session' => $mySession,
                'my_result' => $myResult,
            ];
        });

        return response()->json($data);
    }

    // Detail peserta ujian per ujian (admin): daftar sesi in-progress & finished beserta info peserta dan kelas
    public function participantsStatus(Request $request, Exam $exam)
    {
        $onlyMeta = filter_var($request->input('only_meta'), FILTER_VALIDATE_BOOLEAN);
        // Eager load relasi untuk mendapatkan info kelas/sekolah/grade peserta
        $base = ExamSession::with(['participant.kelas.school', 'participant.kelas.grade', 'result'])
            ->where('exam_id', $exam->id);

        // Filter cohort: prioritaskan sekolah/jenjang dari ujian, fallback ke subject
        $schoolId = $exam->id_school ?: ($exam->subject?->id_school);
        $gradeId  = $exam->id_grade  ?: ($exam->subject?->id_grade);

        // Tambahan filter eksplisit dari request (dukung kedua penamaan untuk kompatibilitas frontend)
        $filterSchoolId = $request->filled('school_id') ? (int)$request->input('school_id')
            : ($request->filled('id_school') ? (int)$request->input('id_school') : null);
        $filterClassId  = $request->filled('class_id') ? (int)$request->input('class_id')
            : ($request->filled('id_class') ? (int)$request->input('id_class') : null);
        $filterGradeId  = $request->filled('grade_id') ? (int)$request->input('grade_id')
            : ($request->filled('id_grade') ? (int)$request->input('id_grade') : null);

        $search = trim((string) $request->input('search', ''));
        $limit  = (int) $request->input('limit', 100);
        $limit  = max(10, min($limit, 500));

        if ($schoolId || $gradeId || $filterSchoolId || $filterGradeId || $filterClassId) {
            $base = $base->whereHas('participant', function ($qp) use ($schoolId, $gradeId, $filterSchoolId, $filterGradeId, $filterClassId) {
                $qp->whereHas('kelas', function ($qk) use ($schoolId, $gradeId, $filterSchoolId, $filterGradeId, $filterClassId) {
                    if ($schoolId)      $qk->where('id_school', $schoolId);
                    if ($gradeId)       $qk->where('id_grade', $gradeId);
                    if ($filterSchoolId) $qk->where('id_school', $filterSchoolId);
                    if ($filterGradeId)  $qk->where('id_grade', $filterGradeId);
                    if ($filterClassId)  $qk->where('id', $filterClassId);
                });
            });
        }
        if ($search !== '') {
            $base = $base->whereHas('participant', function ($qp) use ($search) {
                $qp->where(function ($w) use ($search) {
                    $w->where('nama', 'like', "%$search%")
                      ->orWhere('nisn', 'like', "%$search%");
                });
            });
        }

        $inProgressTotal = 0;
        $finishedTotal = 0;
        $inProgressSessions = collect();
        $finishedSessions = collect();
        if (!$onlyMeta) {
            $inProgressTotal = (clone $base)
                ->whereNotNull('started_at')
                ->whereNull('finished_at')
                ->count();

            $finishedTotal = (clone $base)
                ->whereNotNull('finished_at')
                ->count();

            $inProgressSessions = (clone $base)
                ->whereNotNull('started_at')
                ->whereNull('finished_at')
                ->orderByDesc('started_at')
                ->limit($limit)
                ->get();

            $finishedSessions = (clone $base)
                ->whereNotNull('finished_at')
                ->orderByDesc('finished_at')
                ->limit($limit)
                ->get();
        }

        $mapSession = function($session) use ($exam) {
            $p = $session->participant;
            $kelas = $p?->kelas;
            $school = $kelas?->school;
            $grade = $kelas?->grade;
            $res = $session->result;

            $start = \Carbon\Carbon::parse($exam->scheduled_at, config('app.timezone'));
            $end = (clone $start)->addMinutes((int)$exam->duration_minutes);
            $now = now();
            $hasResult = (bool) $res;
            $finishRequested = \Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','finish_requested_at') && (bool)$session->finish_requested_at;
            $attempts = \Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','submissions_count') ? (int)$session->submissions_count : 0;
            $expiredNow = $now->gt($end);

            if ($hasResult) {
                $status = 'success';
            } elseif ($attempts >= 3) {
                $status = 'end_pending';
            } elseif ($finishRequested) {
                $status = 'end_pending';
            } elseif ($expiredNow) {
                $status = 'expired';
            } elseif ($session->started_at && !$session->finished_at) {
                $status = 'mulai';
            } else {
                $status = $session->finished_at ? 'selesai' : ($now->lt($start) ? 'belum_mulai' : 'mulai');
            }

            return [
                'session_id' => $session->id,
                'started_at' => optional($session->started_at)->toDateTimeString(),
                'finished_at' => optional($session->finished_at)->toDateTimeString(),
                'finish_requested_at' => optional($session->finish_requested_at)->toDateTimeString(),
                'submissions_count' => $attempts,
                'status' => $status,
                'participant' => [
                    'id' => $p?->id,
                    'nisn' => $p?->nisn,
                    'nama' => $p?->nama,
                    'id_kelas' => $p?->id_kelas,
                    'name_class' => $p?->name_class,
                    'jurusan' => $p?->jurusan,
                    'last_activity' => optional($p?->last_activity)->toDateTimeString(),
                ],
                'class' => [
                    'id' => $kelas?->id,
                    'name' => $kelas?->name,
                    'id_school' => $kelas?->id_school,
                    'id_grade' => $kelas?->id_grade,
                ],
                'school' => [
                    'id' => $school?->id,
                    'nama' => $school?->nama,
                ],
                'grade' => [
                    'id' => $grade?->id,
                    'grade' => $grade?->grade,
                ],
                'result' => $res ? [
                    'score' => $res->score,
                    'correct_count' => $res->correct_count,
                    'wrong_count' => $res->wrong_count,
                    'essay_correct_count' => $res->essay_correct_count,
                    'essay_wrong_count' => $res->essay_wrong_count,
                    'total_count' => $res->total_count,
                    'correct_order_indexes' => $res->correct_order_indexes,
                    'wrong_order_indexes' => $res->wrong_order_indexes,
                    'correct_orders_essays' => $res->correct_orders_essays,
                    'wrong_orders_essay' => $res->wrong_orders_essay,
                    'answers' => $res->answers,
                    'essay_details' => $res->essay_details,
                ] : null,
            ];
        };

        $inProgress = $inProgressSessions->map($mapSession);
        $finished = $finishedSessions->map($mapSession);

        // Tentukan cohort peserta berdasarkan sekolah/jenjang ujian (prioritas ujian, lalu subject) + filter eksplisit request
        $cohortQuery = null;
        $schoolId = $schoolId ?? ($exam->id_school ?: ($exam->subject?->id_school));
        $gradeId  = $gradeId  ?? ($exam->id_grade  ?: ($exam->subject?->id_grade));
        if ($schoolId || $gradeId || $filterSchoolId || $filterGradeId || $filterClassId) {
            $cohortQuery = \App\Models\ExamParticipant::with(['kelas.school', 'kelas.grade'])
                ->whereHas('kelas', function ($q) use ($schoolId, $gradeId, $filterSchoolId, $filterGradeId, $filterClassId) {
                    if ($schoolId)       $q->where('id_school', $schoolId);
                    if ($gradeId)        $q->where('id_grade', $gradeId);
                    if ($filterSchoolId) $q->where('id_school', $filterSchoolId);
                    if ($filterGradeId)  $q->where('id_grade', $filterGradeId);
                    if ($filterClassId)  $q->where('id', $filterClassId);
                });
            if ($search !== '') {
                $cohortQuery = $cohortQuery->where(function ($qq) use ($search) {
                    $qq->where('nama', 'like', "%$search%")
                       ->orWhere('nisn', 'like', "%$search%");
                });
            }
        }

        // Daftar peserta yang belum mulai: tidak memiliki sesi untuk ujian ini
        $notStarted = collect();
        if (!$onlyMeta && $cohortQuery) {
            $startedIds = \App\Models\ExamSession::where('exam_id', $exam->id)->pluck('exam_participant_id')->all();
            $pnq = $cohortQuery->whereNotIn('id', $startedIds);
            $notStartedTotal = (clone $pnq)->count();
            $participantsNotStarted = $pnq->limit($limit)->get();
            $notStarted = $participantsNotStarted->map(function ($p) {
                $kelas = $p?->kelas; $school = $kelas?->school; $grade = $kelas?->grade;
                return [
                    'participant' => [
                        'id' => $p?->id,
                        'nisn' => $p?->nisn,
                        'nama' => $p?->nama,
                        'id_kelas' => $p?->id_kelas,
                        'name_class' => $p?->name_class,
                        'jurusan' => $p?->jurusan,
                        'last_activity' => optional($p?->last_activity)->toDateTimeString(),
                    ],
                    'class' => [
                        'id' => $kelas?->id,
                        'name' => $kelas?->name,
                        'id_school' => $kelas?->id_school,
                        'id_grade' => $kelas?->id_grade,
                    ],
                    'school' => [
                        'id' => $school?->id,
                        'nama' => $school?->nama,
                    ],
                    'grade' => [
                        'id' => $grade?->id,
                        'grade' => $grade?->grade,
                    ],
                ];
            });
        }

        // Peserta tidak aktif (belum login/idle) dalam cohort berdasarkan last_activity
        $thresholdMinutes = 15; // konsisten dengan dashboard
        $inactive = collect();
        $inactiveTotal = 0;
        if (!$onlyMeta && $cohortQuery) {
            $inactiveQuery = (clone $cohortQuery)
                ->where(function($q) use ($thresholdMinutes) {
                    $q->whereNull('last_activity')
                      ->orWhere('last_activity', '<', now()->subMinutes($thresholdMinutes));
                });
            $inactiveTotal = (clone $inactiveQuery)->count();
            $inactiveParticipants = $inactiveQuery->limit($limit)->get();
            $inactive = $inactiveParticipants->map(function ($p) {
                $kelas = $p?->kelas; $school = $kelas?->school; $grade = $kelas?->grade;
                return [
                    'participant' => [
                        'id' => $p?->id,
                        'nisn' => $p?->nisn,
                        'nama' => $p?->nama,
                        'id_kelas' => $p?->id_kelas,
                        'name_class' => $p?->name_class,
                        'jurusan' => $p?->jurusan,
                        'last_activity' => optional($p?->last_activity)->toDateTimeString(),
                    ],
                    'class' => [
                        'id' => $kelas?->id,
                        'name' => $kelas?->name,
                        'id_school' => $kelas?->id_school,
                        'id_grade' => $kelas?->id_grade,
                    ],
                    'school' => [
                        'id' => $school?->id,
                        'nama' => $school?->nama,
                    ],
                    'grade' => [
                        'id' => $grade?->id,
                        'grade' => $grade?->grade,
                    ],
                ];
            });
        }

        // Kumpulkan daftar kelas unik untuk dropdown filter (tanpa batasan limit)
        $distinctClassIds = [];
        try {
            // Kelas dari sesi yang ada untuk ujian ini
            $sessParticipantIds = \App\Models\ExamSession::where('exam_id', $exam->id)->pluck('exam_participant_id')->all();
            if (!empty($sessParticipantIds)) {
                $classIdsFromSess = \App\Models\ExamParticipant::whereIn('id', $sessParticipantIds)->pluck('id_kelas')->all();
                foreach ($classIdsFromSess as $cid) { if ($cid) $distinctClassIds[$cid] = true; }
            }
            // Kelas dari cohort peserta (sesuai filter sekolah/grade), tanpa limit
            if ($cohortQuery) {
                $classIdsFromCohort = (clone $cohortQuery)->pluck('id_kelas')->all();
                foreach ($classIdsFromCohort as $cid) { if ($cid) $distinctClassIds[$cid] = true; }
            }
        } catch (\Throwable $e) {
            // Abaikan kegagalan pengumpulan kelas unik agar endpoint tetap berjalan
        }

        $classesList = [];
        try {
            $classIds = array_keys($distinctClassIds);
            if (!empty($classIds)) {
                $classes = \App\Models\SchoolClass::whereIn('id', $classIds)->orderBy('name','asc')->get(['id','name']);
                $classesList = $classes->map(function($c){ return ['id' => (int)$c->id, 'name' => (string)$c->name]; })->values();
            }
        } catch (\Throwable $e) {
            $classesList = [];
        }

        // Override daftar kelas saat only_meta: ambil penuh dari sekolah terpilih atau sekolah ujian
        if ($onlyMeta) {
            try {
                $schoolForList = $filterSchoolId ?: ($exam->id_school ?: ($exam->subject?->id_school));
                $gradeForList = $filterGradeId ?: ($exam->id_grade ?: ($exam->subject?->id_grade));
                if ($schoolForList) {
                    $classes = \App\Models\SchoolClass::query()
                        ->where('id_school', $schoolForList)
                        ->when($gradeForList, function($q) use ($gradeForList) { $q->where('id_grade', $gradeForList); })
                        ->orderBy('name','asc')
                        ->get(['id','name']);
                    $classesList = $classes->map(function($c){ return ['id' => (int)$c->id, 'name' => (string)$c->name]; })->values();
                }
            } catch (\Throwable $e) { /* ignore */ }
        }

        // Kumpulkan daftar sekolah unik untuk dropdown
        $schoolsList = [];
        try {
            $classIds = array_keys($distinctClassIds);
            if (!empty($classIds)) {
                $classRows = \App\Models\SchoolClass::whereIn('id', $classIds)->get(['id','id_school']);
                $schoolIds = [];
                foreach ($classRows as $cr) { if ($cr->id_school) $schoolIds[$cr->id_school] = true; }
                $ids = array_keys($schoolIds);
                if (!empty($ids)) {
                    $schools = \App\Models\School::whereIn('id', $ids)->orderBy('nama','asc')->get(['id','nama']);
                    $schoolsList = $schools->map(function($s){ return ['id' => (int)$s->id, 'nama' => (string)$s->nama]; })->values();
                }
            }
        } catch (\Throwable $e) {
            $schoolsList = [];
        }

        // Fallback: jika daftar sekolah kosong, masukkan sekolah ujian (bila ada)
        if (empty($schoolsList)) {
            try {
                $sid = $filterSchoolId ?: ($exam->id_school ?: ($exam->subject?->id_school));
                if ($sid) {
                    $s = \App\Models\School::find((int)$sid);
                    if ($s) {
                        $schoolsList = [ ['id' => (int)$s->id, 'nama' => (string)$s->nama] ];
                    }
                }
            } catch (\Throwable $e) { /* ignore */ }
        }

        // Jika hanya metadata diminta, kembalikan tanpa data berat
        if ($onlyMeta) {
            return response()->json([
                'exam' => [
                    'id' => $exam->id,
                    'name' => $exam->name,
                    'code' => $exam->code,
                    'scheduled_at' => $exam->scheduled_at,
                    'id_subject' => $exam->id_subject,
                    'subject_name' => $exam->subject?->name,
                    'subject_code' => $exam->subject?->code,
                ],
                'counts' => [
                    'in_progress' => 0,
                    'finished' => 0,
                    'not_started' => 0,
                    'inactive' => 0,
                ],
                'totals' => [
                    'in_progress' => 0,
                    'finished' => 0,
                    'not_started' => 0,
                    'inactive' => 0,
                ],
                'filters' => [
                    'school_id' => $filterSchoolId,
                    'class_id' => $filterClassId,
                    'grade_id' => $filterGradeId,
                    'search' => $search,
                ],
                'limit' => $limit,
                'classes' => $classesList,
                'schools' => $schoolsList,
                'in_progress' => [],
                'finished' => [],
                'not_started' => [],
                'inactive' => [],
            ]);
        }

        return response()->json([
            'exam' => [
                'id' => $exam->id,
                'name' => $exam->name,
                'code' => $exam->code,
                'scheduled_at' => $exam->scheduled_at,
                'id_subject' => $exam->id_subject,
                'subject_name' => $exam->subject?->name,
                'subject_code' => $exam->subject?->code,
            ],
            'counts' => [
                'in_progress' => $inProgress->count(),
                'finished' => $finished->count(),
                'not_started' => $notStarted->count(),
                'inactive' => $inactive->count(),
            ],
            'totals' => [
                'in_progress' => $inProgressTotal ?? 0,
                'finished' => $finishedTotal ?? 0,
                'not_started' => $notStartedTotal ?? 0,
                'inactive' => $inactiveTotal ?? 0,
            ],
            // pantulkan filter agar frontend dapat men-trace kondisi perhitungan
            'filters' => [
                'school_id' => $filterSchoolId,
                'class_id' => $filterClassId,
                'grade_id' => $filterGradeId,
                'search' => $search,
            ],
            'limit' => $limit,
            // daftar kelas unik untuk dropdown filter
            'classes' => $classesList,
            // daftar sekolah unik untuk dropdown filter
            'schools' => $schoolsList,
            'in_progress' => $inProgress,
            'finished' => $finished,
            'not_started' => $notStarted,
            'inactive' => $inactive,
        ]);
    }

    // Daftar peserta untuk approval (guru/pengawas): sort A-Z, pagination, search sekolah/kelas/nama
    public function approvals(Request $request, Exam $exam)
    {
        $schoolId = $exam->id_school ?: ($exam->subject?->id_school);
        $gradeId  = $exam->id_grade ?: ($exam->subject?->id_grade);

        $perPage = (int) $request->input('per_page', 10);
        $perPage = max(10, min($perPage, 100));
        $page    = (int) $request->input('page', 1);
        $search  = trim((string) $request->input('search', ''));
        $filterSchoolId = $request->filled('school_id') ? (int)$request->input('school_id') : null;
        $filterClassId  = $request->filled('class_id') ? (int)$request->input('class_id') : null;

        $base = \App\Models\ExamParticipant::with(['kelas.school','kelas.grade'])
            ->when($schoolId, function($q) use ($schoolId) {
                $q->whereHas('kelas', fn($k) => $k->where('id_school', $schoolId));
            })
            ->when($gradeId, function($q) use ($gradeId) {
                $q->whereHas('kelas', fn($k) => $k->where('id_grade', $gradeId));
            })
            ->when($filterSchoolId, function($q) use ($filterSchoolId) {
                $q->whereHas('kelas', fn($k) => $k->where('id_school', $filterSchoolId));
            })
            ->when($filterClassId, function($q) use ($filterClassId) {
                $q->whereHas('kelas', fn($k) => $k->where('id', $filterClassId));
            })
            ->when($search !== '', function($q) use ($search) {
                $q->where(function($w) use ($search) {
                    $w->where('nama','like',"%$search%")
                      ->orWhere('nisn','like',"%$search%");
                });
            })
            ->orderBy('nama','asc');

        $paginator = $base->paginate($perPage, ['*'], 'page', $page);
        $participants = collect($paginator->items());

        $ids = $participants->pluck('id')->all();
        $sessions = \App\Models\ExamSession::query()
            ->where('exam_id', $exam->id)
            ->whereIn('exam_participant_id', $ids)
            ->get()
            ->keyBy('exam_participant_id');

        $data = $participants->map(function($p) use ($sessions) {
            $kelas = $p->kelas; $school = $kelas?->school; $grade = $kelas?->grade;
            $sess = $sessions->get($p->id);
            return [
                'participant' => [
                    'id' => (int) $p->id,
                    'nisn' => $p->nisn,
                    'nama' => $p->nama,
                ],
                'school' => $school ? ['id' => (int)$school->id, 'nama' => $school->nama] : null,
                'class' => $kelas ? ['id' => (int)$kelas->id, 'name' => $kelas->name] : null,
                'session' => $sess ? [
                    'id' => (int) $sess->id,
                    'approved_at' => optional($sess->approved_at)->toDateTimeString(),
                    'approved_by_user_id' => $sess->approved_by_user_id ? (int)$sess->approved_by_user_id : null,
                    'started_at' => optional($sess->started_at)->toDateTimeString(),
                    'finished_at' => optional($sess->finished_at)->toDateTimeString(),
                ] : null,
            ];
        })->values();

        return response()->json([
            'data' => $data,
            'current_page' => $paginator->currentPage(),
            'per_page' => $paginator->perPage(),
            'total' => $paginator->total(),
            'last_page' => $paginator->lastPage(),
        ]);
    }

    // Approve peserta untuk ujian tertentu: buat sesi bila belum ada, set approved_at
    public function approveParticipant(Request $request, Exam $exam, \App\Models\ExamParticipant $exam_participant)
    {
        $user = $request->user();
        if (!($user instanceof \App\Models\User)) {
            return response()->json(['message' => 'Hanya admin/guru yang dapat meng-approve peserta'], 403);
        }
        // Hanya izinkan approve jika sesi peserta SUDAH ada (artinya peserta sudah login/memulai proses)
        $session = \App\Models\ExamSession::query()
            ->where('exam_id', $exam->id)
            ->where('exam_participant_id', $exam_participant->id)
            ->first();

        if (!$session) {
            return response()->json([
                'message' => 'Peserta belum login atau belum memulai ujian. Tidak dapat di-approve.',
            ], 422);
        }

        $session->approved_at = now();
        $session->approved_by_user_id = (int) $user->id;
        $session->save();

        return response()->json([
            'message' => 'Peserta di-approve untuk ujian ini',
            'session_id' => (int) $session->id,
            'approved_at' => $session->approved_at->toDateTimeString(),
        ]);
    }
    // Mulai ujian untuk peserta: buat sesi dan urutan soal yang deterministik per peserta
    public function startForParticipant(Request $request, Exam $exam)
    {
        $user = $request->user();
        if (!($user instanceof \App\Models\ExamParticipant)) {
            return response()->json(['message' => 'Hanya peserta ujian yang dapat memulai ujian'], 403);
        }

        // Pastikan ujian dalam rentang waktu
        $start = Carbon::parse($exam->scheduled_at, config('app.timezone'));
        $end = (clone $start)->addMinutes((int)$exam->duration_minutes);
        if (!now()->between($start, $end)) {
            return response()->json(['message' => 'Ujian belum/telah berakhir'], 422);
        }

        // Ambil semua id soal berdasarkan subject ujian
        $questionIds = Question::where('subject_id', $exam->id_subject)->pluck('id')->all();
        // Jika belum ada soal
        if (empty($questionIds)) {
            return response()->json(['message' => 'Belum ada soal untuk ujian ini'], 422);
        }

        // Urutan deterministik per peserta-ujian menggunakan hash crc32
        $score = fn($qid) => sprintf('%u', crc32($user->id.'-'.$exam->id.'-'.$qid));
        usort($questionIds, function($a, $b) use ($score) {
            $sa = $score($a); $sb = $score($b);
            if ($sa == $sb) return 0;
            return $sa < $sb ? -1 : 1;
        });

        // Buat/ambil sesi TANPA mulai jika belum di-approve
        $session = ExamSession::firstOrCreate(
            ['exam_id' => $exam->id, 'exam_participant_id' => $user->id],
            ['question_order' => $questionIds]
        );
        if (!$session->question_order || !is_array($session->question_order)) {
            $session->question_order = $questionIds;
            $session->save();
        }

        // Wajib approved oleh guru/pengawas sebelum boleh mulai
        if (!$session->approved_at) {
            return response()->json(['message' => 'Belum di-approve. Mohon tunggu persetujuan pengawas.'], 422);
        }

        // Increment entry_attempts saat peserta kembali masuk sesi yang sama
        $queued = false;
        if (!$session->wasRecentlyCreated) {
            if (\Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','entry_attempts')) {
                $session->entry_attempts = (int) $session->entry_attempts + 1;
            }
            if (!$session->started_at) { $session->started_at = now(); }
            $session->save();

            // Jika attempts > 3 dan sesi belum selesai, tandai finish dan antrekan worker
            $hasEntryCol = \Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','entry_attempts');
            $overLimit = $hasEntryCol ? ((int)$session->entry_attempts > 3) : false;
            $alreadyFinished = (bool) $session->finished_at;
            $finishRequested = \Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','finish_requested_at') && (bool)$session->finish_requested_at;
            if ($overLimit && !$alreadyFinished && !$finishRequested) {
                if (\Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','finish_requested_at')) {
                    $session->finish_requested_at = now();
                    $session->save();
                }
                \App\Jobs\ProcessFinalExamSubmission::dispatch(
                    (int) $session->id,
                    (int) $user->id,
                    is_array($session->draft_answers) ? $session->draft_answers : [],
                    'attempts_exhausted'
                );
                $queued = true;
            }
        }

        return response()->json([
            'session_id' => $session->id,
            'exam' => [
                'id' => $exam->id,
                'name' => $exam->name,
                'code' => $exam->code,
                'duration_minutes' => $exam->duration_minutes,
                'scheduled_at' => $exam->scheduled_at,
            ],
            'question_order' => $session->question_order,
        ]);
    }

    // Detail sesi ujian: kembalikan daftar soal sesuai urutan tersimpan
    public function sessionDetail(Request $request, ExamSession $session)
    {
        $user = $request->user();
        if (!($user instanceof \App\Models\ExamParticipant)) {
            return response()->json(['message' => 'Hanya peserta ujian yang dapat mengakses sesi'], 403);
        }
        if ($session->exam_participant_id !== $user->id) {
            return response()->json(['message' => 'Sesi ini bukan milik Anda'], 403);
        }

        $exam = $session->exam()->first();
        $ids = is_array($session->question_order) ? $session->question_order : [];
        if (empty($ids)) {
            return response()->json(['message' => 'Urutan soal tidak tersedia'], 422);
        }

        // Cek apakah ujian sedang berlangsung untuk menentukan apakah perlu caching
        $start = Carbon::parse($exam->scheduled_at, config('app.timezone'));
        $end = (clone $start)->addMinutes((int)$exam->duration_minutes);
        $isExamActive = now()->between($start, $end);

        $questions = [];
        $cacheKey = "exam_questions_{$exam->id}_subject_{$exam->id_subject}";

        // Bila sudah lewat waktu ujian, hapus cache agar tidak tersimpan terus
        if (now()->gt($end)) {
            Cache::forget($cacheKey);
        }
        
        if ($isExamActive) {
            // Cache soal ujian hanya saat ujian sedang berlangsung
            $cacheTtl = max(1, $end->diffInMinutes(now()) + 5); // Cache sampai ujian selesai + 5 menit buffer
            
            $questions = Cache::remember($cacheKey, $cacheTtl * 60, function () use ($ids) {
                return Question::whereIn('id', $ids)
                    ->get(['id','text','option_a','option_b','option_c','option_d','option_e','image_path'])
                    ->keyBy('id')
                    ->toArray();
            });
            
            // Convert array kembali ke collection untuk konsistensi
            $questions = collect($questions);
        } else {
            // Jika ujian tidak aktif, ambil langsung dari database tanpa cache
            $questions = Question::whereIn('id', $ids)
                ->get(['id','text','option_a','option_b','option_c','option_d','option_e','image_path'])
                ->keyBy('id');
        }

        // Susun soal sesuai urutan yang tersimpan di sesi
        $ordered = [];
        foreach ($ids as $qid) {
            if (isset($questions[$qid])) {
                $ordered[] = is_array($questions[$qid]) ? (object)$questions[$qid] : $questions[$qid];
            }
        }

        return response()->json([
            'session_id' => $session->id,
            'exam' => [
                'id' => $exam->id,
                'name' => $exam->name,
                'code' => $exam->code,
                'duration_minutes' => $exam->duration_minutes,
                'scheduled_at' => $exam->scheduled_at,
            ],
            'questions' => $ordered,
            'draft_answers' => is_array($session->draft_answers) ? $session->draft_answers : [],
        ]);
    }

    // Admin: reset entry_attempts peserta untuk ujian tertentu (kembalikan ke 1)
    public function adminResetEntryAttempts(Request $request, Exam $exam, \App\Models\ExamParticipant $exam_participant)
    {
        $user = $request->user();
        if (!($user instanceof \App\Models\User)) {
            return response()->json(['message' => 'Hanya admin/guru yang dapat mereset kesempatan masuk'], 403);
        }

        $session = \App\Models\ExamSession::query()
            ->where('exam_id', $exam->id)
            ->where('exam_participant_id', $exam_participant->id)
            ->first();

        if (!$session) {
            return response()->json(['message' => 'Sesi peserta tidak ditemukan untuk ujian ini'], 404);
        }

        // Pastikan kolom entry_attempts tersedia
        $hasEntryCol = \Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','entry_attempts');
        if (!$hasEntryCol) {
            return response()->json(['message' => 'Fitur kesempatan masuk tidak diaktifkan pada database'], 422);
        }

        // Reset ke 1, dan hapus permintaan finish jika ada agar peserta bisa lanjut
        $session->entry_attempts = 1;
        if (\Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','finish_requested_at')) {
            $session->finish_requested_at = null;
        }
        $session->save();

        return response()->json([
            'message' => 'Kesempatan masuk peserta direset menjadi 1',
            'session_id' => (int) $session->id,
            'entry_attempts' => (int) $session->entry_attempts,
        ]);
    }

    // Autosave jawaban peserta ke queue: simpan ke draft_answers tanpa evaluasi
    public function autosaveAnswers(Request $request, ExamSession $session)
    {
        $user = $request->user();
        if (!($user instanceof \App\Models\ExamParticipant)) {
            return response()->json(['message' => 'Hanya peserta ujian yang dapat menyimpan jawaban'], 403);
        }
        if ($session->exam_participant_id !== $user->id) {
            return response()->json(['message' => 'Sesi ini bukan milik Anda'], 403);
        }

        $request->validate([
            'answers' => 'required|array',
            // Terima pilihan A-E untuk pilihan ganda dan teks bebas untuk essay
            'answers.*' => 'nullable|string|max:5000',
        ]);

        $answersInput = $request->input('answers', []);
        if (!is_array($answersInput)) {
            return response()->json(['message' => 'Format jawaban tidak valid'], 422);
        }

        // Dispatch ke queue untuk mengurangi beban request
        \App\Jobs\SaveExamDraftAnswers::dispatch(
            $session->id,
            $user->id,
            $answersInput,
            [
                'ip' => $request->ip(),
                'ua' => (string)($request->header('User-Agent') ?? ''),
            ]
        );

        return response()->json(['message' => 'Autosave queued']);
    }

    // Tandai sesi peserta selesai (finish) — enqueue finalize via worker, tanpa update finished_at langsung
    public function finishSession(Request $request, ExamSession $session)
    {
        $user = $request->user();
        if (!($user instanceof \App\Models\ExamParticipant)) {
            return response()->json(['message' => 'Hanya peserta ujian yang dapat menyelesaikan sesi'], 403);
        }
        if ($session->exam_participant_id !== $user->id) {
            return response()->json(['message' => 'Sesi ini bukan milik Anda'], 403);
        }

        // Catat permintaan selesai tanpa menyentuh finished_at (menghindari write massal)
        if (\Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','finish_requested_at')) {
            $session->finish_requested_at = now();
            $session->save();
        }

        // Antrekan job finalisasi; hasil dibuat oleh worker (atau inline jika QUEUE_CONNECTION=sync)
        $reason = $request->input('reason');
        \App\Jobs\ProcessFinalExamSubmission::dispatch(
            $session->id,
            $user->id,
            is_array($session->draft_answers) ? $session->draft_answers : [],
            $reason
        );

        if ($reason) {
            \Illuminate\Support\Facades\Log::info('Exam session finish requested', [
                'session_id' => $session->id,
                'participant_id' => $user->id,
                'reason' => $reason,
                'finish_requested_at' => optional($session->finish_requested_at)->toDateTimeString(),
            ]);
        }

        return response()->json([
            'status' => 'queued',
            'message' => 'Finalisasi sesi diantre untuk diproses',
            'queued' => true,
        ], 202);
    }

    // Laporkan pelanggaran perilaku ujian (tanpa menyimpan ke DB khusus, log ke server)
    public function reportViolation(Request $request, ExamSession $session)
    {
        $user = $request->user();
        if (!($user instanceof \App\Models\ExamParticipant)) {
            return response()->json(['message' => 'Hanya peserta ujian yang dapat melaporkan pelanggaran terkait sesi mereka'], 403);
        }
        if ($session->exam_participant_id !== $user->id) {
            return response()->json(['message' => 'Sesi ini bukan milik Anda'], 403);
        }
        $request->validate([
            'type' => 'required|string|max:50',
            'detail' => 'nullable|array',
        ]);

        $type = (string) $request->input('type', 'unknown');
        $detail = $request->input('detail', []);
        $detail = is_array($detail) ? $detail : ['raw' => $detail];

        Log::warning('Exam violation detected', [
            'session_id' => $session->id,
            'participant_id' => $user->id,
            'type' => $type,
            'detail' => $detail,
            'timestamp' => now()->toDateTimeString(),
        ]);

        return response()->json(['message' => 'Violation logged']);
    }

    // Submit jawaban peserta: hitung nilai berdasar urutan kanonik mata pelajaran, simpan hasil
    public function submitAnswers(Request $request, ExamSession $session)
    {
        $user = $request->user();
        if (!($user instanceof \App\Models\ExamParticipant)) {
            return response()->json(['message' => 'Hanya peserta ujian yang dapat mengirim jawaban'], 403);
        }
        if ($session->exam_participant_id !== $user->id) {
            return response()->json(['message' => 'Sesi ini bukan milik Anda'], 403);
        }

        $request->validate([
            'answers' => 'required|array',
            // Terima pilihan A-E untuk pilihan ganda dan teks bebas untuk essay
            'answers.*' => 'nullable|string|max:5000',
            'reason' => 'nullable|string|max:100'
        ]);

        $answersInput = $request->input('answers', []);
        if (!is_array($answersInput)) {
            return response()->json(['message' => 'Format jawaban tidak valid'], 422);
        }

        // Simpan draft_answers secara sinkron sebagai fallback (mirip autosave)
        $current = is_array($session->draft_answers) ? $session->draft_answers : [];
        $merged = $current;
        foreach ($answersInput as $qid => $opt) {
            $qidInt = (int) $qid;
            if ($qidInt <= 0) continue;

            if ($opt === null) { unset($merged[$qidInt]); continue; }
            if (!is_string($opt)) { continue; }

            $raw = (string) $opt;
            $trimmed = trim($raw);
            if ($trimmed === '') { unset($merged[$qidInt]); continue; }

            $upper = strtoupper($trimmed);
            if (in_array($upper, ['A','B','C','D','E'], true)) {
                $merged[$qidInt] = $upper;
            } else {
                $merged[$qidInt] = $trimmed;
            }
        }
        $session->draft_answers = $merged;
        $session->save();

        // Tandai permintaan selesai agar UI menampilkan End Pending
        if (\Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','finish_requested_at')) {
            $session->finish_requested_at = now();
            $session->save();
        }

        $reason = $request->input('reason');

        // Antrekan job untuk proses final submit
        ProcessFinalExamSubmission::dispatch(
            $session->id,
            $user->id,
            $answersInput,
            $reason
        );

        if ($reason) {
            \Illuminate\Support\Facades\Log::info('Exam answers submit queued', [
                'session_id' => $session->id,
                'participant_id' => $user->id,
                'reason' => $reason,
            ]);
        }

        return response()->json([
            'status' => 'queued',
            'message' => 'Jawaban diantre untuk diproses',
            'queued' => true,
        ], 202);
    }

    // Superadmin: finalisasi sesi ujian — antrekan job finalize agar diproses oleh worker
    public function adminFinalizeSessions(Request $request, Exam $exam)
    {
        // Validasi input opsional
        $validated = $request->validate([
            'session_ids' => 'nullable|array',
            'session_ids.*' => 'integer',
            'only_in_progress' => 'nullable|boolean',
            'dry_run' => 'nullable|boolean',
        ]);

        $onlyInProgress = array_key_exists('only_in_progress', $validated)
            ? (bool)$validated['only_in_progress']
            : true; // default: hanya sesi yang belum selesai
        $dryRun = array_key_exists('dry_run', $validated) ? (bool)$validated['dry_run'] : false;

        $query = ExamSession::query()->where('exam_id', $exam->id);
        if (!empty($validated['session_ids'])) {
            $query->whereIn('id', $validated['session_ids']);
        }
        if ($onlyInProgress) {
            $query->whereNull('finished_at');
        }

        $sessions = $query->get();

        $stats = [
            'sessions_count' => $sessions->count(),
            'queued' => 0,
        ];

        foreach ($sessions as $session) {
            $answers = is_array($session->draft_answers) ? $session->draft_answers : [];
            if (!$dryRun) {
                \App\Jobs\ProcessFinalExamSubmission::dispatch(
                    (int) $session->id,
                    (int) $session->exam_participant_id,
                    (array) $answers,
                    'admin_finalize'
                );
                // catat permintaan finalisasi tanpa menyentuh finished_at
                if (\Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','finish_requested_at')) {
                    $session->finish_requested_at = now();
                    $session->save();
                }
            }
            $stats['queued']++;
        }

        return response()->json([
            'stats' => $stats,
            'queued_to' => 'finalize',
            'dry_run' => $dryRun,
        ], $dryRun ? 200 : 202);
    }

        // Tambahan: tokenisasi dengan frekuensi (mengabaikan stopwords) untuk Cosine Similarity
        private function wordTokens(string $text): array
        {
            $normalized = $this->normalizeText($text);
            $parts = preg_split('/\s+/u', $normalized, -1, PREG_SPLIT_NO_EMPTY) ?: [];
            $stop = [
                'dan','yang','di','ke','dari','untuk','dengan','atau','pada','sebagai','adalah','itu','ini','kami','kita','saya','aku','anda','para','serta','hingga','agar','karena','namun','bahwa','jadi','dalam','akan','atau','tidak','ya','bukan','oleh','sebuah','suatu','sebuah','terhadap','lebih','kurang','lain','juga'
            ];
            $stopSet = array_fill_keys($stop, true);
            $freq = [];
            foreach ($parts as $p) {
                if (mb_strlen($p, 'UTF-8') < 2) continue;
                if (isset($stopSet[$p])) continue;
                $freq[$p] = ($freq[$p] ?? 0) + 1;
            }
            return $freq; // token => count
        }

        // Cosine similarity (0..1) berbasis frekuensi token
        private function cosineSimilarity(string $a, string $b): float
        {
            $fa = $this->wordTokens($a);
            $fb = $this->wordTokens($b);
            if (empty($fa) || empty($fb)) return 0.0;
            $dot = 0.0; $normA = 0.0; $normB = 0.0;
            foreach ($fa as $t => $ca) {
                $normA += $ca * $ca;
                if (isset($fb[$t])) { $dot += $ca * $fb[$t]; }
            }
            foreach ($fb as $cb) { $normB += $cb * $cb; }
            $den = sqrt($normA) * sqrt($normB);
            if ($den <= 0.0) return 0.0;
            return $dot / $den;
        }

        // Rabin-Karp substring match pada teks yang sudah dinormalisasi
        private function rabinKarpContains(string $text, string $pattern): bool
        {
            $t = $this->normalizeText($text);
            $p = $this->normalizeText($pattern);
            $n = strlen($t);
            $m = strlen($p);
            if ($m === 0) return true;
            if ($m > $n) return false;

            $d = 256; $q = 101; // basis dan modulus kecil
            $h = 1;
            for ($i = 0; $i < $m - 1; $i++) { $h = ($h * $d) % $q; }
            $pHash = 0; $tHash = 0;
            for ($i = 0; $i < $m; $i++) {
                $pHash = ($d * $pHash + ord($p[$i])) % $q;
                $tHash = ($d * $tHash + ord($t[$i])) % $q;
            }
            for ($i = 0; $i <= $n - $m; $i++) {
                if ($pHash === $tHash) {
                    // verifikasi untuk menghindari collision
                    $substr = substr($t, $i, $m);
                    if ($substr === $p) return true;
                }
                if ($i < $n - $m) {
                    $tHash = ($d * ($tHash - ord($t[$i]) * $h) + ord($t[$i + $m])) % $q;
                    if ($tHash < 0) $tHash += $q;
                }
            }
            return false;
        }
    public function adminResetParticipantSession(\Illuminate\Http\Request $request, \App\Models\Exam $exam, \App\Models\ExamParticipant $examParticipant)
    {
        // Hanya admin melalui middleware permission yang bisa akses
        return \Illuminate\Support\Facades\DB::transaction(function () use ($request, $exam, $examParticipant) {
            $session = \App\Models\ExamSession::where('exam_id', $exam->id)
                ->where('exam_participant_id', $examParticipant->id)
                ->first();

            if (!$session) {
                return response()->json(['message' => 'Sesi peserta untuk ujian ini tidak ditemukan'], 404);
            }

            $force = (bool) $request->boolean('force', false);
            $inProgress = (bool) ($session->started_at && !$session->finished_at);
            if ($inProgress && !$force) {
                return response()->json(['message' => 'Sesi masih berjalan. Gunakan force=true untuk reset paksa.'], 422);
            }

            // Hapus hasil terkait sesi jika ada
            \App\Models\ExamResult::where('exam_session_id', $session->id)->delete();
            // Hapus sesi
            $session->delete();

            return response()->json(['message' => 'Sesi peserta direset. Peserta dapat memulai ujian lagi.']);
        });
    }
    public function exportResults(\Illuminate\Http\Request $request, \App\Models\Exam $exam)
    {
        // Ambil semua sesi dan hasilnya untuk ujian ini
        $sessions = \App\Models\ExamSession::with(['participant.kelas.school','participant.kelas.grade','result'])
            ->where('exam_id', $exam->id)
            ->orderByDesc('finished_at')
            ->get();

        $headers = [
            'Exam ID', 'Exam Name', 'Subject',
            'Session ID', 'Started At', 'Finished At',
            'Participant ID', 'NISN', 'Nama', 'Kelas', 'Sekolah', 'Jenjang',
            'Correct Count', 'Wrong Count', 'Essay Correct', 'Essay Wrong', 'Total Count', 'Score',
            'Correct Order Indexes', 'Wrong Order Indexes', 'Correct Orders Essays', 'Wrong Orders Essay',
            'Jawaban (mapel urutan kanonik)'
        ];

        // Siapkan writer CSV ke memori (dengan BOM agar Excel nyaman)
        $stream = fopen('php://temp', 'r+');
        // Tulis BOM UTF-8
        fwrite($stream, "\xEF\xBB\xBF");
        fputcsv($stream, $headers);

        foreach ($sessions as $s) {
            $p = $s->participant;
            $kelas = $p?->kelas;
            $school = $kelas?->school;
            $grade = $kelas?->grade;
            $res = $s->result;

            // Pastikan tidak ada kolom kosong: gunakan '-' untuk string kosong/null, 0 untuk angka
            $str = function ($v) { $t = trim((string)($v ?? '')); return $t !== '' ? $t : '-'; };
            $num = function ($v) { $n = (int)($v ?? 0); return $n; };

            // Tentukan urutan kanonik soal: prioritaskan question_order pada sesi, fallback ke subject
            $order = is_array($s->question_order) ? array_map('intval', $s->question_order) : [];
            if (empty($order)) {
                $subjectId = (int)($exam->id_subject ?? 0);
                if ($subjectId > 0) {
                    $order = \App\Models\Question::where('subject_id', $subjectId)
                        ->orderBy('created_at','asc')->orderBy('id','asc')
                        ->pluck('id')->map(fn($id) => (int)$id)->all();
                }
            }

            // Ambil jawaban dari hasil; jika kosong, fallback draft_answers sesi
            $answersArr = [];
            if ($res && is_array($res->answers)) { $answersArr = $res->answers; }
            elseif (is_array($s->draft_answers)) { $answersArr = $s->draft_answers; }

            // Susun jawaban mengikuti urutan kanonik; kosong menjadi '-'
            $ordered = [];
            foreach ($order as $qid) {
                $raw = $answersArr[$qid] ?? null;
                $ans = $str($raw);
                $ordered[] = $ans;
            }
            // Jika tidak ada order, tetap gabungkan array jawaban dengan sort kunci agar deterministik
            if (empty($ordered) && !empty($answersArr)) {
                ksort($answersArr, SORT_NUMERIC);
                foreach ($answersArr as $raw) { $ordered[] = $str($raw); }
            }
            // Pastikan ada isi (minimal '-') meski tidak ada jawaban sama sekali
            if (empty($ordered)) { $ordered = ['-']; }

            $answersCanonical = implode('|', $ordered);

            // Konversi array indeks urutan ke string aman
            $joinInts = function ($arr) {
                if (!is_array($arr) || empty($arr)) return '-';
                $ints = array_map(fn($v) => (int)$v, $arr);
                return implode('|', $ints);
            };

            // Hitung nilai total termasuk Essay (tanpa mengubah nilai persisten)
            $mcqCorrect = (int)($res?->correct_count ?? 0);
            $essayCorrect = (int)($res?->essay_correct_count ?? 0);
            $totalCount = (int)($res?->total_count ?? 0);
            $scoreTotal = 0.0;
            if ($totalCount > 0) {
                $scoreTotal = (($mcqCorrect + $essayCorrect) / $totalCount) * 100.0;
                // Bulatkan 2 desimal
                $scoreTotal = round($scoreTotal, 2);
            }

            fputcsv($stream, [
                $exam->id,
                $str($exam->name),
                $str($exam->subject?->name ?? ''),
                $s->id,
                $str(optional($s->started_at)->toDateTimeString()),
                $str(optional($s->finished_at)->toDateTimeString()),
                $p?->id ?? 0,
                $str($p?->nisn ?? ''),
                $str($p?->nama ?? ''),
                $str($kelas?->name ?? ''),
                $str($school?->nama ?? ''),
                $str($grade?->grade ?? ''),
                $num($mcqCorrect),
                $num($res?->wrong_count ?? 0),
                $num($totalCount),
                $num($res?->score ?? 0),
                $scoreTotal,
                $joinInts($res?->correct_order_indexes ?? []),
                $joinInts($res?->wrong_order_indexes ?? []),
                $joinInts($res?->correct_orders_essays ?? []),
                $joinInts($res?->wrong_orders_essay ?? []),
                $answersCanonical,
            ]);
        }

        rewind($stream);
        $csvContent = stream_get_contents($stream);
        fclose($stream);

        $filename = 'exam_'.$exam->id.'_results_'.date('Ymd_His').'.csv';
        return response($csvContent, 200, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"'
        ]);
    }
    public function exportItemAnalysis(\Illuminate\Http\Request $request, \App\Models\Exam $exam)
    {
        // Ambil sesi yang selesai beserta peserta, kelas, dan hasil
        $sessions = \App\Models\ExamSession::with(['participant.kelas.school','participant.kelas.grade','result'])
            ->where('exam_id', $exam->id)
            ->whereNotNull('finished_at')
            ->get();

        // Ambil urutan kanonik soal berdasarkan subject (ID ASC sesuai permintaan)
        $canonicalQuestions = \App\Models\Question::where('subject_id', $exam->id_subject)
            ->orderBy('id','asc')
            ->get(['id','key_answer','option_b','option_c','option_d']);
        $questionCount = $canonicalQuestions->count();
        if ($questionCount === 0) {
            return response()->json(['message' => 'Tidak ada soal untuk mata pelajaran ini'], 422);
        }

        // Helper normalisasi teks essay
        $normalizeText = function ($t) {
            $s = trim((string)($t ?? ''));
            $noTags = preg_replace('/<[^>]*>/', '', $s);
            $uni = preg_replace('/\s+/u', ' ', $noTags);
            return mb_strtolower(trim($uni ?? ''), 'UTF-8');
        };

        // Urutkan baris berdasarkan nama kelas lalu nama peserta
        $sessions = $sessions->sort(function ($a, $b) {
            $ka = optional($a->participant?->kelas)->name ?? '';
            $kb = optional($b->participant?->kelas)->name ?? '';
            $na = $a->participant?->nama ?? '';
            $nb = $b->participant?->nama ?? '';
            if ($ka === $kb) return strcmp($na, $nb);
            return strcmp($ka, $kb);
        });

        // Siapkan Spreadsheet XLSX
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header: NO, NISN, NAMA, KELAS, MAPEL, NAMA UJIAN, nomor-nomor soal, Nilai Pilihan, Nilai Essay
         $headers = ['NO','NISN','NAMA','KELAS','MAPEL','NAMA UJIAN'];
         for ($i = 1; $i <= $questionCount; $i++) { $headers[] = (string)$i; }
         $headers[] = 'Nilai Pilihan';
         $headers[] = 'Nilai Essay';

        // Tulis header ke baris 1
        foreach ($headers as $idx => $label) {
            $colLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($idx + 1);
            $sheet->setCellValue($colLetter . '1', $label);
        }

        // Tulis data baris
        $seq = 1; $rowIndex = 2;
        foreach ($sessions as $s) {
            $p = $s->participant; $kelas = $p?->kelas; $res = $s->result;
            $subjectName = $exam->subject?->name ?? '';
            $examName = $exam->name ?? '';

            // Kolom dasar
            $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(1) . $rowIndex, $seq);
            $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(2) . $rowIndex, (string)($p?->nisn ?? ''));
            $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(3) . $rowIndex, (string)($p?->nama ?? ''));
            $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(4) . $rowIndex, (string)($kelas?->name ?? ''));
            $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(5) . $rowIndex, (string)$subjectName);
            $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(6) . $rowIndex, (string)$examName);

            // Pakai draft_answers dari exam_sessions; fallback ke answers hasil jika kosong
            $answers = is_array($s->draft_answers) ? $s->draft_answers : (is_array($res?->answers) ? $res->answers : []);
            $essayDetails = is_array($res?->essay_details) ? $res->essay_details : [];

            // Tulis jawaban per nomor 1..N sesuai urutan kanonik, dan beri warna hijau untuk yang benar
            $questionStartCol = 7;
            foreach ($canonicalQuestions as $qIndex => $q) {
                $qid = (string)$q->id;
                $val = $answers[$qid] ?? null;

                // Deteksi essay: essay jika key bukan A–E dan opsi B–D kosong
                $isBlankOrDash = function ($v) { $t = trim((string)$v); return $t === '' || $t === '-'; };
                $keyUp = strtoupper(trim((string)$q->key_answer));
                $keyIsMC = in_array($keyUp, ['A','B','C','D','E'], true);
                $bdBlank = $isBlankOrDash($q->option_b) && $isBlankOrDash($q->option_c) && $isBlankOrDash($q->option_d);
                $isEssay = (!$keyIsMC) && $bdBlank;

                $isAnswered = ($val !== null && trim((string)$val) !== '');
                $isCorrect = false; $valOut = '';

                if ($isAnswered) {
                    if ($isEssay) {
                        $detail = $essayDetails[$qid] ?? null;
                        if ($detail && isset($detail['status'])) {
                            $isCorrect = ($detail['status'] === 'correct');
                        } else {
                            $normAns = $normalizeText($val);
                            $normKey = $normalizeText($q->key_answer ?? '');
                            $isCorrect = ($normAns !== '' && $normAns === $normKey);
                        }
                        $valOut = $isCorrect ? '✓' : '✗';
                    } else {
                        $choice = strtoupper(trim((string)$val));
                        $isCorrect = ($choice !== '' && $keyUp !== '' && $choice === $keyUp);
                        $valOut = $choice ?: '-';
                    }
                } else {
                    $valOut = '-';
                }

                $col = $questionStartCol + $qIndex;
                $colLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col);
                $sheet->setCellValue($colLetter . $rowIndex, $valOut);
                if ($isAnswered && $isCorrect) {
                    $sheet->getStyle($colLetter . $rowIndex)
                        ->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                        ->getStartColor()->setRGB('C6EFCE');
                }
            }

            // Nilai Pilihan: ambil dari score di ExamResult (berdasarkan NISN)
            $nilaiPilihan = (float)($res?->score ?? 0.0);
            $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($questionStartCol + $questionCount) . $rowIndex, $nilaiPilihan);

            // Nilai Essay: correct_essays / (correct_essays + wrong_essays) * 100
            $correctEssays = is_array($res?->correct_orders_essays) ? count($res->correct_orders_essays) : 0;
            $wrongEssays = is_array($res?->wrong_orders_essay) ? count($res->wrong_orders_essay) : 0;
            $essayTotalCalc = $correctEssays + $wrongEssays;
            $nilaiEssay = $essayTotalCalc > 0 ? round(($correctEssays / $essayTotalCalc) * 100, 2) : 0.0;
            $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($questionStartCol + $questionCount + 1) . $rowIndex, $nilaiEssay);

            $seq++; $rowIndex++;
        }

        // Auto-size kolom
        for ($c = 1; $c <= count($headers); $c++) {
            $sheet->getColumnDimension(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($c))->setAutoSize(true);
        }

        // Tulis XLSX ke output
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        ob_start();
        $writer->save('php://output');
        $xlsxContent = ob_get_clean();

        $filename = 'analisis_butir_soal_exam_'.$exam->id.'_'.date('Ymd_His').'.xlsx';
        return response($xlsxContent, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"'
        ]);
    }
}