<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Spatie\Permission\Models\Role;

class AuthController extends Controller
{
    public function login(Request $request)
    {
        $request->validate(['email'=>'required|email','password'=>'required']);
        $user = User::where('email',$request->email)->first();

        if(!$user) return response()->json(['message'=>'Email tidak terdaftar'],404);
        if(!Hash::check($request->password,$user->password)) return response()->json(['message'=>'Email atau password salah'],401);
        // Blokir login jika status pengguna "Tidak Aktif"
        if (strtolower($user->status ?? '') === 'tidak aktif') {
            return response()->json(['message' => 'Akun tidak aktif'], 403);
        }

        // Update aktivitas terakhir saat login sukses
        $user->last_activity = now();
        $user->save();

        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'user' => $user->load('roles'),
            'permissions' => $user->getAllPermissions()->pluck('name'),
            'token' => $token
        ]);
    }

    // Registrasi publik: hanya role guru/murid, status default "Tidak Aktif"
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|min:3',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6|confirmed',
            'role' => 'required|in:guru,murid',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'status' => 'Tidak Aktif',
        ]);

        // Assign role terbatas
        $role = Role::where('name', $request->role)->first();
        if ($role) {
            $user->assignRole($role);
        }

        return response()->json([
            'message' => 'Registrasi berhasil. Akun belum aktif, menunggu persetujuan admin.',
            'user' => $user->load('roles'),
        ], 201);
    }

    public function logout(Request $request)
    {
        $user = $request->user();
        $user->tokens()->delete();
        // Jangan ubah status aktivasi; cukup kosongkan last_activity
        $user->update(['last_activity' => null]);
        return response()->json(['message'=>'Logout berhasil']);
    }

    public function me(Request $request)
    {
        $user = $request->user();
        // Hitung status online secara dinamis tanpa menulis ke kolom status
        $isOnline = $user->last_activity && now()->diffInMinutes($user->last_activity) <= 15;
        return response()->json([
            'user' => $user->load('roles'),
            'is_online' => $isOnline,
            'permissions' => $user->getAllPermissions()->pluck('name')
        ]);
    }

    public function updateActivity(Request $request){
        $user = $request->user();
        $user->last_activity = now();
        $user->save();
        return response()->json(['message'=>'Aktivitas diperbarui']);
    }

    // Ubah password untuk user yang sedang login (self-service)
    public function changePassword(Request $request)
    {
        $user = $request->user();
        $request->validate([
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:6|confirmed',
        ]);

        if (!Hash::check($request->input('current_password'), $user->password)) {
            return response()->json(['message' => 'Password saat ini salah'], 422);
        }

        $user->password = Hash::make($request->input('new_password'));
        $user->save();

        return response()->json(['message' => 'Password berhasil diubah']);
    }
}
