<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\Models\Exam;
use App\Models\ExamSession;
use App\Jobs\ProcessFinalExamSubmission;

class FinalizeExpiredExamSessions extends Command
{
    protected $signature = 'exams:finalize-expired {--exam-id=* : Restrict to specific exam ids} {--limit=500 : Max sessions to queue per run} {--dry-run : Do not dispatch jobs, just print summary}';
    protected $description = 'Queue finalization for in-progress sessions of exams that have expired.';

    public function handle(): int
    {
        $now = Carbon::now(config('app.timezone'));
        $limit = max(1, (int) $this->option('limit'));
        $examIdsOpt = (array) $this->option('exam-id');
        $dryRun = (bool) $this->option('dry-run');

        $this->info(sprintf('[%s] Scanning expired exams...', $now->toDateTimeString()));

        $examQuery = Exam::query()
            ->whereNotNull('scheduled_at')
            ->where('duration_minutes', '>', 0);
        if (!empty($examIdsOpt)) {
            $examIdsOpt = array_map('intval', $examIdsOpt);
            $examQuery->whereIn('id', $examIdsOpt);
        }

        $expiredExamIds = [];
        $examQuery->orderBy('scheduled_at', 'asc')->chunk(200, function ($chunk) use (&$expiredExamIds, $now) {
            foreach ($chunk as $exam) {
                // Hitung waktu berakhir
                $start = Carbon::parse($exam->scheduled_at, config('app.timezone'));
                $end = (clone $start)->addMinutes((int) $exam->duration_minutes);
                if ($end->lte($now)) {
                    $expiredExamIds[] = (int) $exam->id;
                }
            }
        });

        if (empty($expiredExamIds)) {
            $this->info('No expired exams found.');
            return self::SUCCESS;
        }

        $totalQueued = 0;
        $totalSessions = 0;
        $queuedPerExam = [];

        foreach ($expiredExamIds as $examId) {
            $sessionsQuery = ExamSession::query()
                ->where('exam_id', $examId)
                ->whereNotNull('started_at')
                ->whereNull('finished_at')
                ->orderBy('started_at', 'asc');

            $count = 0;
            $sessionsQuery->chunk(200, function ($chunk) use (&$count, &$totalSessions, &$totalQueued, $limit, $dryRun) {
                foreach ($chunk as $session) {
                    $totalSessions++;
                    if ($totalQueued >= $limit) {
                        break; // respect global limit per run
                    }
                    $answers = is_array($session->draft_answers) ? $session->draft_answers : [];
                    if ($dryRun) {
                        $count++;
                        $totalQueued++;
                        continue;
                    }
                    ProcessFinalExamSubmission::dispatch(
                        (int) $session->id,
                        (int) $session->exam_participant_id,
                        (array) $answers,
                        'scheduler_expired'
                    );
                    $count++;
                    $totalQueued++;
                }
            });
            $queuedPerExam[$examId] = $count;
            if ($totalQueued >= $limit) {
                break; // stop after reaching limit
            }
        }

        $summary = [
            'expired_exams' => count($expiredExamIds),
            'sessions_in_progress' => $totalSessions,
            'queued' => $totalQueued,
            'limit' => $limit,
            'dry_run' => $dryRun,
            'queued_per_exam' => $queuedPerExam,
            'timestamp' => $now->toDateTimeString(),
        ];

        if (!$dryRun) {
            Log::info('Auto finalize expired exams queued', $summary);
        }

        $this->line(json_encode($summary));
        return self::SUCCESS;
    }
}